# TypeScript Project Template

This is a TypeScript project configured for AI-assisted development with TDD and DDD principles.

## 📋 Quick Start

### Prerequisites

- Node.js 18+ installed
- npm or yarn or pnpm
- Git configured

### Setup

```bash
# Create new project
mkdir myproject
cd myproject
npm init -y

# Install TypeScript
npm install -D typescript @types/node

# Install testing tools
npm install -D jest @types/jest ts-jest

# Install linting tools
npm install -D eslint @typescript-eslint/parser @typescript-eslint/eslint-plugin
npm install -D prettier

# Initialize TypeScript
npx tsc --init

# Initialize Jest
npx ts-jest config:init

# Build
npm run build

# Run tests
npm test
```

## 🏗️ Project Structure

```
myproject/
├── src/
│   ├── domain/                # Domain models
│   │   ├── user.ts
│   │   └── order.ts
│   ├── repository/            # Data access
│   │   └── userRepository.ts
│   ├── service/               # Business logic
│   │   └── userService.ts
│   └── index.ts               # Entry point
├── tests/                     # Tests
│   ├── unit/
│   │   └── user.test.ts
│   └── integration/
│       └── userService.test.ts
├── dist/                      # Compiled output
├── package.json
├── tsconfig.json              # TypeScript config
├── jest.config.js             # Jest config
├── .eslintrc.js               # ESLint config
├── .prettierrc                # Prettier config
├── .gitignore
├── AGENTS_RULES.md            # AI agent rules (READ THIS FIRST)
├── AGENTS.md                  # Quick reference
├── CODE_QUALITY_PRINCIPLES.md
├── CONTRIBUTING.md
└── README.md                  # This file
```

## 🧪 Testing

### Run Tests

```bash
# Run all tests
npm test

# Run with coverage
npm run test:coverage

# Run in watch mode
npm run test:watch

# Run specific test
npm test -- user.test.ts
```

### Test Example

```typescript
// tests/unit/user.test.ts
import { User } from '../../src/domain/user';

describe('User', () => {
  it('should create a valid user', () => {
    const user = new User('John', 'john@example.com');
    
    expect(user.name).toBe('John');
    expect(user.email).toBe('john@example.com');
  });

  it('should throw error for invalid email', () => {
    expect(() => {
      new User('John', 'invalid-email');
    }).toThrow('Invalid email');
  });

  it('should validate user', () => {
    const user = new User('John', 'john@example.com');
    expect(user.isValid()).toBe(true);
  });
});
```

## 🔧 Development

### TypeScript Configuration

```json
// tsconfig.json
{
  "compilerOptions": {
    "target": "ES2022",
    "module": "commonjs",
    "lib": ["ES2022"],
    "outDir": "./dist",
    "rootDir": "./src",
    "strict": true,
    "noImplicitAny": true,
    "strictNullChecks": true,
    "strictFunctionTypes": true,
    "noUnusedLocals": true,
    "noUnusedParameters": true,
    "esModuleInterop": true,
    "skipLibCheck": true,
    "forceConsistentCasingInFileNames": true
  },
  "include": ["src/**/*"],
  "exclude": ["node_modules", "dist", "tests"]
}
```

### Build

```bash
# Compile TypeScript
tsc

# Watch mode
tsc --watch

# Build with npm script
npm run build
```

### Code Quality

```bash
# Lint code
npm run lint

# Fix linting issues
npm run lint:fix

# Format code
npm run format

# Type check
npm run type-check
```

## 📦 Package.json Scripts

```json
{
  "scripts": {
    "build": "tsc",
    "dev": "ts-node src/index.ts",
    "test": "jest",
    "test:coverage": "jest --coverage",
    "test:watch": "jest --watch",
    "lint": "eslint src/**/*.ts",
    "lint:fix": "eslint src/**/*.ts --fix",
    "format": "prettier --write \"src/**/*.ts\"",
    "type-check": "tsc --noEmit"
  }
}
```

## 🎯 Example: User Domain

### Domain Model

```typescript
// src/domain/user.ts
export interface IUser {
  readonly id: string;
  readonly name: string;
  readonly email: string;
}

export class User implements IUser {
  readonly id: string;
  readonly name: string;
  readonly email: string;

  constructor(name: string, email: string) {
    if (!name || name.trim().length === 0) {
      throw new Error('Name cannot be empty');
    }
    
    if (!email.includes('@')) {
      throw new Error('Invalid email');
    }

    this.id = crypto.randomUUID();
    this.name = name;
    this.email = email;
  }

  isValid(): boolean {
    return this.name.length > 0 && this.email.includes('@');
  }
}
```

### Result Pattern

```typescript
// src/types/result.ts
export type Result<T, E> = 
  | { success: true; value: T }
  | { success: false; error: E };

export function ok<T>(value: T): Result<T, never> {
  return { success: true, value };
}

export function err<E>(error: E): Result<never, E> {
  return { success: false, error };
}
```

### Repository Interface

```typescript
// src/domain/userRepository.ts
import { User } from './user';
import { Result } from '../types/result';

export interface UserRepository {
  findById(id: string): Promise<Result<User, string>>;
  findByEmail(email: string): Promise<Result<User, string>>;
  save(user: User): Promise<Result<void, string>>;
  delete(id: string): Promise<Result<void, string>>;
}
```

### Service

```typescript
// src/service/userService.ts
import { User } from '../domain/user';
import { UserRepository } from '../domain/userRepository';
import { Result, ok, err } from '../types/result';

export class UserService {
  constructor(private readonly repository: UserRepository) {}

  async registerUser(
    name: string, 
    email: string
  ): Promise<Result<User, string>> {
    try {
      const user = new User(name, email);
      const saveResult = await this.repository.save(user);
      
      if (!saveResult.success) {
        return err(saveResult.error);
      }
      
      return ok(user);
    } catch (error) {
      return err(error instanceof Error ? error.message : 'Unknown error');
    }
  }
}
```

## 📚 Documentation

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (READ THIS FIRST)
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines

## 🔗 Resources

### Official
- [TypeScript Handbook](https://www.typescriptlang.org/docs/handbook/intro.html)
- [TypeScript Deep Dive](https://basarat.gitbook.io/typescript/)
- [Jest Documentation](https://jestjs.io/docs/getting-started)

### Tools
- [ts-node](https://typestrong.org/ts-node/)
- [ESLint](https://eslint.org/)
- [Prettier](https://prettier.io/)

## 🤝 Contributing

See [CONTRIBUTING.md](./CONTRIBUTING.md) for detailed guidelines.

Quick summary:
1. Read [AGENTS_RULES.md](./AGENTS_RULES.md) completely
2. Follow TDD: Write tests first
3. Maintain 90% code coverage
4. Never use `any` type
5. Use strict TypeScript config
6. Use conventional commit format
7. Get approval before committing

## 📄 License

[Your License Here]
