# Contributing to TypeScript Projects

## Rules

### Architecture

Preserve the architecture but propose enhancement if needed by asking with examples by indicating pros and cons.

### Domain Driven Design / Acceptance Tests

As a software engineer, it is important to separate "technical" vs "business" rules. Domain Driven Design is a technique to start first with "business" problems and vocabulary but requires "technical" code (algorithm, patterns, ...) to run.

Generally **Acceptance Tests** are focused on domain as input and output (test setup, data and assertion) while **Unit Test** is focusing on testing something simple (business or technical).

### The "Red" Phase in TDD

Design matters. Sometimes, it is better to start with a **top-down approach** which consists of implementing the design by using stub implementations; the input and output of functions must be used though. The code should compile while tests will stay red but it is an important step to validate the design first.

### Example: Top-Down Design in TypeScript

```typescript
// Step 1: Define the high-level function with proper types
export function processOrder(order: Order): Result<ProcessedOrder, OrderError> {
  const validated = validateOrder(order);
  if (!validated.success) return validated;
  
  const priced = calculatePricing(validated.value);
  if (!priced.success) return priced;
  
  const saved = saveOrder(priced.value);
  return saved;
}

// Step 2: Stub implementations
function validateOrder(order: Order): Result<Order, OrderError> {
  // TODO(agt): Implement order validation
  throw new Error('Not implemented');
}

function calculatePricing(order: Order): Result<Order, OrderError> {
  // TODO(agt): Implement pricing calculation
  throw new Error('Not implemented');
}

function saveOrder(order: Order): Result<ProcessedOrder, OrderError> {
  // TODO(agt): Implement order persistence
  throw new Error('Not implemented');
}
```

## Test-Driven Development Workflow

### 1. Red Phase: Write a Failing Test

```typescript
// tests/unit/userService.test.ts
import { UserService } from '../../src/service/userService';
import { MockUserRepository } from '../mocks/mockUserRepository';

describe('UserService', () => {
  it('should register a user', async () => {
    const repo = new MockUserRepository();
    const service = new UserService(repo);
    
    const result = await service.registerUser('John', 'john@example.com');
    
    expect(result.success).toBe(true);
    if (result.success) {
      expect(result.value.name).toBe('John');
      expect(result.value.email).toBe('john@example.com');
    }
  });
});
```

### 2. Green Phase: Make it Pass

```typescript
// src/service/userService.ts
import { User } from '../domain/user';
import { UserRepository } from '../domain/userRepository';
import { Result, ok, err } from '../types/result';

export class UserService {
  constructor(private readonly repository: UserRepository) {}
  
  async registerUser(
    name: string, 
    email: string
  ): Promise<Result<User, string>> {
    try {
      const user = new User(name, email);
      const saveResult = await this.repository.save(user);
      
      if (!saveResult.success) {
        return err(saveResult.error);
      }
      
      return ok(user);
    } catch (error) {
      return err(error instanceof Error ? error.message : 'Unknown error');
    }
  }
}
```

### 3. Refactor Phase: Improve the Code

```typescript
export class UserService {
  constructor(private readonly repository: UserRepository) {}
  
  async registerUser(
    name: string, 
    email: string
  ): Promise<Result<User, string>> {
    // Extract validation
    const validationError = this.validateInput(name, email);
    if (validationError) {
      return err(validationError);
    }
    
    try {
      const user = new User(name, email);
      return await this.saveUser(user);
    } catch (error) {
      return err(error instanceof Error ? error.message : 'Unknown error');
    }
  }
  
  private validateInput(name: string, email: string): string | null {
    if (!name || name.trim().length === 0) {
      return 'Name cannot be empty';
    }
    if (!email.includes('@')) {
      return 'Invalid email';
    }
    return null;
  }
  
  private async saveUser(user: User): Promise<Result<User, string>> {
    const saveResult = await this.repository.save(user);
    
    if (!saveResult.success) {
      return err(saveResult.error);
    }
    
    return ok(user);
  }
}
```

## Testing Guidelines

### Unit Tests

```typescript
describe('User', () => {
  it('should create a valid user', () => {
    const user = new User('John', 'john@example.com');
    
    expect(user.name).toBe('John');
    expect(user.email).toBe('john@example.com');
    expect(user.id).toBeDefined();
  });

  it('should throw error for empty name', () => {
    expect(() => {
      new User('', 'john@example.com');
    }).toThrow('Name cannot be empty');
  });

  it('should throw error for invalid email', () => {
    expect(() => {
      new User('John', 'invalid');
    }).toThrow('Invalid email');
  });
});
```

### Integration Tests

```typescript
// tests/integration/userService.test.ts
import { UserService } from '../../src/service/userService';
import { InMemoryUserRepository } from '../../src/repository/inMemoryUserRepository';

describe('UserService Integration', () => {
  it('should register and retrieve a user', async () => {
    const repo = new InMemoryUserRepository();
    const service = new UserService(repo);
    
    const registerResult = await service.registerUser('John', 'john@example.com');
    expect(registerResult.success).toBe(true);
    
    if (registerResult.success) {
      const findResult = await service.getUserById(registerResult.value.id);
      expect(findResult.success).toBe(true);
      
      if (findResult.success) {
        expect(findResult.value.name).toBe('John');
      }
    }
  });
});
```

### Test Coverage

Aim for **90% code coverage** minimum:

```bash
# Run with coverage
npm run test:coverage

# View HTML report
open coverage/lcov-report/index.html
```

## Git Workflow

### Commit Message Format

```
<type>(agt): <description>

Types: feat, fix, refac, chore, doc, test
```

**Examples:**
```
feat(agt): add user authentication module

Implement JWT-based authentication with type-safe token handling.
Includes token generation, validation, and refresh logic.
```

```
fix(agt): resolve type inference issue in user service

Fixed generic type constraint to properly infer return types
from repository methods.
```

### Git Rules

**⚠️ CRITICAL:**
- ❌ **NEVER** use `git stash`
- ❌ **NEVER** use `git push` or `git push --force`
- ✅ **ALWAYS** ask user approval before committing

## Code Review Checklist

- [ ] All tests pass
- [ ] Code coverage is at least 90%
- [ ] No ESLint warnings
- [ ] Code is formatted (Prettier)
- [ ] No `any` types used
- [ ] All functions have explicit return types
- [ ] Proper error handling with Result type
- [ ] Documentation comments for public APIs
- [ ] Commit message follows format

## Best Practices Summary

1. **TDD**: Write tests first (Red-Green-Refactor)
2. **DDD**: Separate business and technical concerns
3. **Type Safety**: Strict mode, no `any`, explicit types
4. **Result Pattern**: Use discriminated unions
5. **Immutability**: Use `readonly` by default
6. **Generics**: For reusable type-safe code
7. **Type Guards**: For runtime type checking

## Resources

- [AGENTS_RULES.md](./AGENTS_RULES.md) - Complete AI agent rules
- [TypeScript Handbook](https://www.typescriptlang.org/docs/handbook/intro.html)
- [Effective TypeScript](https://effectivetypescript.com/)

---

**Remember**: Read [AGENTS_RULES.md](./AGENTS_RULES.md) completely before starting any work.
