# Code Quality Principles for TypeScript

**⚠️ CRITICAL: Read these principles before code reviews and refactoring**

## 1. Never Use `any` Type

### ❌ **Problem**: `any` disables type checking

```typescript
// BAD: Using any
function processData(data: any) {
  return data.value; // No type safety
}

// BAD: Implicit any
function calculate(x, y) { // Parameters have implicit any
  return x + y;
}
```

### ✅ **Solution**: Use `unknown` with type guards

```typescript
// GOOD: Using unknown with type guard
function processData(data: unknown): string {
  if (isValidData(data)) {
    return data.value; // Type-safe
  }
  throw new Error('Invalid data');
}

function isValidData(value: unknown): value is { value: string } {
  return (
    typeof value === 'object' &&
    value !== null &&
    'value' in value &&
    typeof (value as { value: unknown }).value === 'string'
  );
}

// GOOD: Explicit types
function calculate(x: number, y: number): number {
  return x + y;
}
```

### **Why it matters**:
- ❌ `any` defeats the purpose of TypeScript
- ❌ No IntelliSense or autocomplete
- ❌ Runtime errors not caught at compile time
- ✅ `unknown` forces type checking
- ✅ Type guards provide safety

## 2. Explicit Return Types

### ❌ **Problem**: Implicit return types

```typescript
// BAD: Implicit return type
function calculateTotal(items: Item[]) {
  return items.reduce((sum, item) => sum + item.price, 0);
}

// BAD: Inconsistent return
function findUser(id: string) {
  if (id === '123') {
    return { name: 'John' };
  }
  return null; // Inferred as { name: string } | null
}
```

### ✅ **Solution**: Always specify return types

```typescript
// GOOD: Explicit return type
function calculateTotal(items: Item[]): number {
  return items.reduce((sum, item) => sum + item.price, 0);
}

// GOOD: Clear return type
function findUser(id: string): User | null {
  if (id === '123') {
    return { id: '123', name: 'John', email: 'john@example.com' };
  }
  return null;
}
```

### **Why it matters**:
- ✅ Self-documenting code
- ✅ Catches return type errors early
- ✅ Better IntelliSense
- ✅ Prevents accidental type changes

## 3. Use Discriminated Unions (Result Pattern)

### ❌ **Problem**: Throwing exceptions

```typescript
// BAD: Throwing exceptions
function parseUser(data: string): User {
  const parsed = JSON.parse(data); // Can throw
  if (!parsed.name) {
    throw new Error('Missing name'); // Exception
  }
  return parsed;
}

// BAD: Nullable return without context
function findUser(id: string): User | null {
  // Lost error information
  return users.find(u => u.id === id) ?? null;
}
```

### ✅ **Solution**: Discriminated unions

```typescript
// GOOD: Result type
type Result<T, E> = 
  | { success: true; value: T }
  | { success: false; error: E };

function parseUser(data: string): Result<User, string> {
  try {
    const parsed = JSON.parse(data);
    if (!parsed.name) {
      return { success: false, error: 'Missing name' };
    }
    return { success: true, value: parsed };
  } catch (error) {
    return { 
      success: false, 
      error: error instanceof Error ? error.message : 'Parse error' 
    };
  }
}

// GOOD: Explicit error handling
function findUser(id: string): Result<User, string> {
  const user = users.find(u => u.id === id);
  
  if (!user) {
    return { success: false, error: `User ${id} not found` };
  }
  
  return { success: true, value: user };
}

// Usage with type narrowing
const result = findUser("123");
if (result.success) {
  console.log(result.value.name); // Type-safe
} else {
  console.error(result.error); // Type-safe
}
```

### **Why it matters**:
- ✅ No hidden exceptions
- ✅ Explicit error handling
- ✅ Type-safe error information
- ✅ Compiler enforces handling

## 4. Never Use Non-Null Assertion (`!`)

### ❌ **Problem**: Non-null assertion bypasses type safety

```typescript
// BAD: Non-null assertion
const user = findUser("123")!; // Assumes not null
console.log(user.name); // Can crash if null

// BAD: Multiple assertions
const value = data!.user!.name!; // Very dangerous
```

### ✅ **Solution**: Use type guards or optional chaining

```typescript
// GOOD: Type guard
const user = findUser("123");
if (user !== null) {
  console.log(user.name); // Type-safe
}

// GOOD: Optional chaining
const name = data?.user?.name ?? 'Unknown';

// GOOD: Early return
function processUser(id: string): void {
  const user = findUser(id);
  if (user === null) {
    return;
  }
  
  console.log(user.name); // Type-safe
}
```

### **Why it matters**:
- ❌ `!` can cause runtime crashes
- ❌ Defeats null safety
- ✅ Type guards are safe
- ✅ Optional chaining is concise

## 5. Use Readonly for Immutability

### ❌ **Problem**: Mutable state

```typescript
// BAD: Mutable interface
interface User {
  id: string;
  name: string;
  email: string;
}

const user: User = createUser();
user.name = 'Changed'; // Can be modified
```

### ✅ **Solution**: Use readonly

```typescript
// GOOD: Readonly interface
interface User {
  readonly id: string;
  readonly name: string;
  readonly email: string;
}

const user: User = createUser();
// user.name = 'Changed'; // Error: readonly

// GOOD: Readonly array
function processItems(items: readonly Item[]): number {
  // items.push(newItem); // Error: readonly
  return items.length;
}

// GOOD: Const assertion
const CONFIG = {
  apiUrl: 'https://api.example.com',
  timeout: 5000,
} as const;

// CONFIG.timeout = 3000; // Error: readonly
```

### **Why it matters**:
- ✅ Prevents accidental mutations
- ✅ Easier to reason about code
- ✅ Enables optimizations
- ✅ Functional programming style

## 6. Use Generics for Reusability

### ❌ **Problem**: Code duplication

```typescript
// BAD: Duplicated code
function findUserById(id: string): User | null {
  return users.find(u => u.id === id) ?? null;
}

function findOrderById(id: string): Order | null {
  return orders.find(o => o.id === id) ?? null;
}

function findProductById(id: string): Product | null {
  return products.find(p => p.id === id) ?? null;
}
```

### ✅ **Solution**: Generic function

```typescript
// GOOD: Generic function
function findById<T extends { id: string }>(
  items: T[],
  id: string
): T | null {
  return items.find(item => item.id === id) ?? null;
}

// Usage
const user = findById(users, "123");
const order = findById(orders, "456");
const product = findById(products, "789");

// GOOD: Generic class
class Repository<T extends { id: string }> {
  private items: T[] = [];

  add(item: T): void {
    this.items.push(item);
  }

  findById(id: string): T | null {
    return this.items.find(item => item.id === id) ?? null;
  }

  getAll(): readonly T[] {
    return this.items;
  }
}
```

### **Why it matters**:
- ✅ DRY (Don't Repeat Yourself)
- ✅ Type-safe reusability
- ✅ Easier maintenance
- ✅ Better abstractions

## 7. Testing Quality

### Use Type-Safe Tests

```typescript
describe('UserService', () => {
  let service: UserService;
  let mockRepo: jest.Mocked<UserRepository>;

  beforeEach(() => {
    mockRepo = {
      findById: jest.fn(),
      save: jest.fn(),
    } as jest.Mocked<UserRepository>;
    
    service = new UserService(mockRepo);
  });

  it('should create a user', async () => {
    mockRepo.save.mockResolvedValue({ success: true, value: undefined });
    
    const result = await service.registerUser('John', 'john@example.com');
    
    expect(result.success).toBe(true);
    if (result.success) {
      expect(result.value.name).toBe('John');
    }
  });
});
```

## Summary

1. **Never `any`**: Use `unknown` with type guards
2. **Explicit Types**: Always specify return types
3. **Result Pattern**: Use discriminated unions
4. **No `!`**: Use type guards or optional chaining
5. **Readonly**: Immutability by default
6. **Generics**: For reusable type-safe code
7. **Testing**: Type-safe tests, 90% coverage

**Remember**: These principles make TypeScript code safe, maintainable, and scalable.
