# AI Agent Rules for TypeScript Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on TypeScript projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file
3. Read the `package.json` for dependencies and scripts

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
npm test
npm run test:coverage
npm run lint
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check coverage report for new/modified code (minimum 90% coverage)
3. **ESLint passes** - No warnings or errors

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Test Organization
- **Unit tests**: In `__tests__` directory or `.test.ts` files
- **Integration tests**: In `tests/integration/`
- Use `describe` and `it` blocks
- Use `expect` assertions from Jest

---

## Development

### TypeScript-Specific Best Practices

#### 1. Strict Type Safety
- **Enable strict mode** in tsconfig.json
- **Never use `any`** - use `unknown` instead
- **Explicit return types** on all functions
- **No implicit any** parameters

```typescript
// tsconfig.json
{
  "compilerOptions": {
    "strict": true,
    "noImplicitAny": true,
    "strictNullChecks": true,
    "strictFunctionTypes": true,
    "noUnusedLocals": true,
    "noUnusedParameters": true
  }
}

// Bad: Using any
function process(data: any) {
  return data.value;
}

// Good: Using unknown with type guard
function process(data: unknown): string {
  if (typeof data === 'object' && data !== null && 'value' in data) {
    return String((data as { value: unknown }).value);
  }
  throw new Error('Invalid data');
}

// Good: Explicit return type
function calculateTotal(items: Item[]): number {
  return items.reduce((sum, item) => sum + item.price, 0);
}
```

#### 2. Interfaces and Type Aliases
- **Use interfaces** for object shapes
- **Use type aliases** for unions, intersections, and complex types
- **Prefer readonly** for immutability

```typescript
// Interface for object shape
interface User {
  readonly id: string;
  readonly name: string;
  readonly email: string;
}

// Type alias for union
type Result<T, E> = 
  | { success: true; value: T }
  | { success: false; error: E };

// Type alias for intersection
type Timestamped<T> = T & {
  readonly createdAt: Date;
  readonly updatedAt: Date;
};

type UserWithTimestamp = Timestamped<User>;
```

#### 3. Discriminated Unions (Result Pattern)
- **Use discriminated unions** for Result/Option types
- **Type-safe error handling**
- **No throwing exceptions** in business logic

```typescript
type Result<T, E> = 
  | { success: true; value: T }
  | { success: false; error: E };

function findUser(id: string): Result<User, string> {
  const user = users.find(u => u.id === id);
  
  if (!user) {
    return { success: false, error: `User ${id} not found` };
  }
  
  return { success: true, value: user };
}

// Usage with type narrowing
const result = findUser("123");
if (result.success) {
  console.log(result.value.name); // Type-safe
} else {
  console.error(result.error); // Type-safe
}
```

#### 4. Generics
- **Use generics** for reusable type-safe code
- **Constrain generics** with extends
- **Infer types** when possible

```typescript
// Generic function with constraint
function findById<T extends { id: string }>(
  items: T[],
  id: string
): T | null {
  return items.find(item => item.id === id) ?? null;
}

// Generic class
class Repository<T extends { id: string }> {
  private items: T[] = [];

  add(item: T): void {
    this.items.push(item);
  }

  findById(id: string): T | null {
    return this.items.find(item => item.id === id) ?? null;
  }

  getAll(): readonly T[] {
    return this.items;
  }
}
```

#### 5. Type Guards
- **Use type guards** for runtime type checking
- **User-defined type guards** with `is` keyword
- **Never use `!` non-null assertion**

```typescript
// Type guard function
function isUser(value: unknown): value is User {
  return (
    typeof value === 'object' &&
    value !== null &&
    'id' in value &&
    'name' in value &&
    'email' in value
  );
}

// Usage
function processData(data: unknown): void {
  if (isUser(data)) {
    console.log(data.name); // Type-safe
  } else {
    throw new Error('Invalid user data');
  }
}

// Bad: Using non-null assertion
const user = findUser("123")!; // DON'T DO THIS

// Good: Using type guard
const user = findUser("123");
if (user !== null) {
  console.log(user.name);
}
```

#### 6. Immutability
- **Use readonly** for properties
- **Use const assertions** for literals
- **Use ReadonlyArray** or readonly modifier

```typescript
// Readonly interface
interface Config {
  readonly apiUrl: string;
  readonly timeout: number;
  readonly retries: number;
}

// Const assertion
const ROLES = ['admin', 'user', 'guest'] as const;
type Role = typeof ROLES[number]; // 'admin' | 'user' | 'guest'

// Readonly array
function processItems(items: readonly Item[]): number {
  // items.push(newItem); // Error: readonly
  return items.length;
}
```

#### 7. Async/Await
- **Use async/await** for asynchronous code
- **Return Result type** instead of throwing
- **Handle errors** explicitly

```typescript
async function fetchUser(id: string): Promise<Result<User, string>> {
  try {
    const response = await fetch(`/api/users/${id}`);
    
    if (!response.ok) {
      return { 
        success: false, 
        error: `HTTP ${response.status}: ${response.statusText}` 
      };
    }
    
    const data = await response.json();
    
    if (!isUser(data)) {
      return { success: false, error: 'Invalid user data' };
    }
    
    return { success: true, value: data };
  } catch (error) {
    return { 
      success: false, 
      error: error instanceof Error ? error.message : 'Unknown error' 
    };
  }
}
```

### Code Quality Principles

#### 1. Never Use `any`
```typescript
// Bad: Using any
function process(data: any) {
  return data.value;
}

// Good: Using unknown with type guard
function process(data: unknown): string {
  if (isValidData(data)) {
    return data.value;
  }
  throw new Error('Invalid data');
}
```

#### 2. Explicit Return Types
```typescript
// Bad: Implicit return type
function calculateTotal(items: Item[]) {
  return items.reduce((sum, item) => sum + item.price, 0);
}

// Good: Explicit return type
function calculateTotal(items: Item[]): number {
  return items.reduce((sum, item) => sum + item.price, 0);
}
```

#### 3. Use Discriminated Unions
```typescript
// Good: Type-safe state management
type LoadingState<T> =
  | { status: 'idle' }
  | { status: 'loading' }
  | { status: 'success'; data: T }
  | { status: 'error'; error: string };

function handleState<T>(state: LoadingState<T>): void {
  switch (state.status) {
    case 'idle':
      console.log('Waiting...');
      break;
    case 'loading':
      console.log('Loading...');
      break;
    case 'success':
      console.log('Data:', state.data); // Type-safe
      break;
    case 'error':
      console.error('Error:', state.error); // Type-safe
      break;
  }
}
```

---

## Commit

### Git Workflow Rules

**⚠️ CRITICAL: Git Rules**

1. **NEVER use `git stash`** - Always commit or discard changes explicitly
2. **NEVER use `git push`** - User will push manually
3. **NEVER use `git push --force`** - Absolutely forbidden
4. **ALWAYS ask user approval** before committing

### Commit Message Format

```
<type>(agt): <description>

Types: feat, fix, refac, chore, doc, test
Example: feat(agt): add user authentication module
```

---

## TypeScript Project Structure

```
myproject/
├── src/
│   ├── domain/
│   │   ├── user.ts
│   │   └── order.ts
│   ├── repository/
│   │   └── userRepository.ts
│   ├── service/
│   │   └── userService.ts
│   └── index.ts
├── tests/
│   ├── unit/
│   └── integration/
├── package.json
├── tsconfig.json
├── jest.config.js
└── README.md
```

---

## Additional Resources

- [TypeScript Handbook](https://www.typescriptlang.org/docs/handbook/intro.html)
- [TypeScript Deep Dive](https://basarat.gitbook.io/typescript/)
- [Effective TypeScript](https://effectivetypescript.com/)

---

**Remember**: This file is your primary reference. Read it completely before starting any work on a TypeScript project.
