# TypeScript AI Agents - Quick Reference

**📖 For complete documentation, see [AGENTS_RULES.md](./AGENTS_RULES.md)**

## Essential Workflow

```
Plan → Test → Develop → Test → Commit
```

1. **Plan**: Discuss with user, create todo-list, get approval
2. **Test**: Write failing tests first (TDD)
3. **Develop**: Implement with TypeScript best practices
4. **Test**: Run `npm test` (90% minimum)
5. **Commit**: Get user approval, use conventional format

## TypeScript Best Practices

### Type Safety
```typescript
// Never use any
// Bad: function process(data: any)
// Good: function process(data: unknown)

// Explicit return types
function calculateTotal(items: Item[]): number {
  return items.reduce((sum, item) => sum + item.price, 0);
}
```

### Interfaces & Types
```typescript
// Interface for object shape
interface User {
  readonly id: string;
  readonly name: string;
  readonly email: string;
}

// Type alias for unions
type Result<T, E> = 
  | { success: true; value: T }
  | { success: false; error: E };
```

### Result Pattern
```typescript
function findUser(id: string): Result<User, string> {
  const user = users.find(u => u.id === id);
  
  if (!user) {
    return { success: false, error: 'User not found' };
  }
  
  return { success: true, value: user };
}

// Usage with type narrowing
const result = findUser("123");
if (result.success) {
  console.log(result.value.name); // Type-safe
}
```

### Generics
```typescript
function findById<T extends { id: string }>(
  items: T[],
  id: string
): T | null {
  return items.find(item => item.id === id) ?? null;
}
```

### Type Guards
```typescript
function isUser(value: unknown): value is User {
  return (
    typeof value === 'object' &&
    value !== null &&
    'id' in value &&
    'name' in value
  );
}

if (isUser(data)) {
  console.log(data.name); // Type-safe
}
```

## Testing

### Unit Tests (Jest)
```typescript
describe('UserService', () => {
  it('should create a user', () => {
    const service = new UserService();
    const user = service.createUser('John', 'john@example.com');
    
    expect(user.name).toBe('John');
    expect(user.email).toBe('john@example.com');
  });

  it('should return error for invalid email', () => {
    const service = new UserService();
    const result = service.createUser('John', 'invalid');
    
    expect(result.success).toBe(false);
  });
});
```

### Run Tests
```bash
# All tests
npm test

# With coverage
npm run test:coverage

# Watch mode
npm run test:watch

# Specific test
npm test -- UserService
```

## Code Quality

### ❌ Don't: Use any or !
```typescript
const value: any = getData(); // BAD
const user = findUser("123")!; // BAD: non-null assertion
```

### ✅ Do: Use unknown and type guards
```typescript
const value: unknown = getData();
if (isValidData(value)) {
  // Type-safe
}

const user = findUser("123");
if (user !== null) {
  console.log(user.name);
}
```

## Git Workflow

### Rules
- ❌ **NEVER** `git stash`
- ❌ **NEVER** `git push` or `--force`
- ✅ **ALWAYS** ask user approval before commit

### Commit Format
```
<type>(agt): <description>

feat(agt): add user authentication
fix(agt): resolve type inference issue
refac(agt): extract validation logic
test(agt): add integration tests
```

## Common Commands

```bash
# Build
tsc                      # Compile
tsc --watch              # Watch mode
npm run build            # Build script

# Testing
npm test                 # Run tests
npm run test:coverage    # With coverage
npm run test:watch       # Watch mode

# Code quality
npm run lint             # ESLint
npm run format           # Prettier
npm run type-check       # Type checking

# Development
npm run dev              # Dev server
ts-node src/index.ts     # Run TypeScript directly
```

## Project Structure

```
myproject/
├── src/
│   ├── domain/
│   ├── repository/
│   ├── service/
│   └── index.ts
├── tests/
│   ├── unit/
│   └── integration/
├── package.json
├── tsconfig.json
└── jest.config.js
```

## Quick Tips

1. **Strict Mode**: Always enable in tsconfig.json
2. **No any**: Use `unknown` instead
3. **Explicit Types**: Always specify return types
4. **Readonly**: Use for immutability
5. **Type Guards**: For runtime type checking
6. **Result Pattern**: For error handling
7. **Generics**: For reusable type-safe code

## Resources

- [AGENTS_RULES.md](./AGENTS_RULES.md) - Complete rules
- [CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md) - Quality guidelines
- [CONTRIBUTING.md](./CONTRIBUTING.md) - Contribution guide
- [TypeScript Handbook](https://www.typescriptlang.org/docs/)
