# Python Project Template

This is a Python project configured for AI-assisted development with TDD and DDD principles.

## 📋 Quick Start

### Prerequisites

- Python 3.10+ installed
- pip or poetry for dependency management
- Git configured

### Setup

```bash
# Create virtual environment
python -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate

# Install dependencies
pip install -e ".[dev]"

# Run tests
pytest tests/

# Type check
mypy src/
```

## 🏗️ Project Structure

```
myproject/
├── src/
│   └── myapp/
│       ├── __init__.py
│       ├── domain/               # Domain models and business logic
│       │   ├── __init__.py
│       │   ├── user.py
│       │   └── order.py
│       ├── repository/           # Data access layer
│       │   ├── __init__.py
│       │   └── user_repository.py
│       ├── service/              # Application services
│       │   ├── __init__.py
│       │   └── user_service.py
│       └── api/                  # API routes
│           ├── __init__.py
│           └── routes.py
├── tests/
│   ├── __init__.py
│   ├── test_user.py
│   └── test_user_service.py
├── pyproject.toml                # Project configuration
├── setup.py or setup.cfg
├── requirements.txt              # Production dependencies
├── requirements-dev.txt          # Development dependencies
├── .gitignore
├── AGENTS_RULES.md               # AI agent rules (READ THIS FIRST)
├── AGENTS.md                     # Quick reference
├── CODE_QUALITY_PRINCIPLES.md
├── CONTRIBUTING.md
└── README.md                     # This file
```

## 🧪 Testing

### Run Tests

```bash
# Run all tests
pytest tests/

# Run with coverage
pytest tests/ --cov=src --cov-report=html

# Run specific test
pytest tests/test_user.py::test_create_user

# Run with markers
pytest -m "not slow"

# Watch mode (requires pytest-watch)
ptw
```

### Test Example

```python
import pytest
from myapp.domain.user import User, create_user, ValidationError

def test_create_user_success():
    user = create_user("John", "john@example.com")
    assert user.name == "John"
    assert user.email == "john@example.com"

def test_create_user_invalid_email():
    with pytest.raises(ValidationError):
        create_user("John", "invalid-email")

@pytest.mark.parametrize("name,email,should_pass", [
    ("John", "john@example.com", True),
    ("", "john@example.com", False),
    ("John", "", False),
    ("John", "invalid", False),
])
def test_create_user_validation(name: str, email: str, should_pass: bool):
    if should_pass:
        user = create_user(name, email)
        assert user.name == name
    else:
        with pytest.raises(ValidationError):
            create_user(name, email)
```

## 🔧 Development

### Type Checking

```bash
# Check types
mypy src/

# Strict mode
mypy --strict src/

# Specific file
mypy src/domain/user.py
```

### Code Formatting

```bash
# Format code
black src/ tests/

# Check formatting
black --check src/ tests/

# Sort imports
isort src/ tests/
```

### Linting

```bash
# Lint with ruff (fast)
ruff check src/ tests/

# Fix auto-fixable issues
ruff check --fix src/ tests/

# Lint with pylint
pylint src/
```

## 📦 Dependencies

### pyproject.toml

```toml
[build-system]
requires = ["setuptools>=45", "wheel"]
build-backend = "setuptools.build_meta"

[project]
name = "myapp"
version = "0.1.0"
requires-python = ">=3.10"
dependencies = [
    "pydantic>=2.0",
    "fastapi>=0.100",
]

[project.optional-dependencies]
dev = [
    "pytest>=7.0",
    "pytest-cov>=4.0",
    "mypy>=1.0",
    "black>=23.0",
    "ruff>=0.1",
    "isort>=5.0",
]
```

## 🎯 Example: User Domain

### Domain Model

```python
# src/myapp/domain/user.py
from dataclasses import dataclass
from typing import Optional
import re

@dataclass(frozen=True)
class User:
    id: str
    name: str
    email: str
    age: Optional[int] = None

class ValidationError(Exception):
    """Raised when validation fails."""
    pass

def create_user(name: str, email: str) -> User:
    """Create a new user with validation."""
    if not name:
        raise ValidationError("Name is required")
    if not _is_valid_email(email):
        raise ValidationError(f"Invalid email: {email}")
    
    return User(
        id=_generate_id(),
        name=name,
        email=email
    )

def _is_valid_email(email: str) -> bool:
    pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
    return bool(re.match(pattern, email))

def _generate_id() -> str:
    import uuid
    return str(uuid.uuid4())
```

### Repository Interface

```python
# src/myapp/domain/user_repository.py
from abc import ABC, abstractmethod
from typing import Optional, List
from .user import User

class UserRepository(ABC):
    @abstractmethod
    def find_by_id(self, user_id: str) -> Optional[User]:
        pass
    
    @abstractmethod
    def find_by_email(self, email: str) -> Optional[User]:
        pass
    
    @abstractmethod
    def save(self, user: User) -> None:
        pass
    
    @abstractmethod
    def delete(self, user_id: str) -> None:
        pass
```

### Service

```python
# src/myapp/service/user_service.py
from typing import Optional
from ..domain.user import User, create_user, ValidationError
from ..domain.user_repository import UserRepository

class UserService:
    def __init__(self, repository: UserRepository):
        self._repository = repository
    
    def register_user(self, name: str, email: str) -> User:
        """Register a new user."""
        # Check if user exists
        existing = self._repository.find_by_email(email)
        if existing:
            raise ValidationError(f"User with email {email} already exists")
        
        # Create and save user
        user = create_user(name, email)
        self._repository.save(user)
        
        return user
    
    def get_user(self, user_id: str) -> Optional[User]:
        """Get user by ID."""
        return self._repository.find_by_id(user_id)
```

## 📚 Documentation

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (READ THIS FIRST)
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines

## 🔗 Resources

### Official
- [Python Documentation](https://docs.python.org/3/)
- [PEP 8 Style Guide](https://peps.python.org/pep-0008/)
- [Type Hints](https://peps.python.org/pep-0484/)

### Tools
- [pytest](https://docs.pytest.org/)
- [mypy](https://mypy.readthedocs.io/)
- [Black](https://black.readthedocs.io/)
- [Ruff](https://docs.astral.sh/ruff/)

## 🤝 Contributing

See [CONTRIBUTING.md](./CONTRIBUTING.md) for detailed guidelines.

Quick summary:
1. Read [AGENTS_RULES.md](./AGENTS_RULES.md) completely
2. Follow TDD: Write tests first
3. Maintain 90% code coverage
4. Use type hints everywhere
5. Format with black, check with mypy
6. Use conventional commit format
7. Get approval before committing

## 📄 License

[Your License Here]
