# Contributing to Python Projects

## Rules

### Architecture

Preserve the architecture but propose enhancement if needed by asking with examples by indicating pros and cons.

**Example of proposing an enhancement:**
```
Current: Services directly call external APIs
Proposed: Add Repository pattern for data access

Pros:
- Better testability (mock repositories)
- Centralized data access logic
- Easier to swap implementations

Cons:
- Additional abstraction layer
- More files to maintain
- Migration effort for existing code

Would you like me to implement this?
```

### Domain Driven Design / Acceptance Tests

As a software engineer, it is important to separate "technical" vs "business" rules. Domain Driven Design is a technique to start first with "business" problems and vocabulary but requires "technical" code (algorithm, patterns, ...) to run.

Generally **Acceptance Tests** are focused on domain as input and output (test setup, data and assertion) while **Unit Test** is focusing on testing something simple (business or technical). 

Acceptance Tests are more robust when refactoring because they are higher level and Input/Output should not change during a refactoring. At some point, implementing a feature requires a refactoring; in that case it is important to **make it work, then make it right** without changing acceptance tests.

### The "Red" Phase in TDD

The `red` part in TDD matters for compile time. Design can be validated quickly before doing implementation then refactoring. When underlying parts are easy to implement, it means that the design is good. Having business vocabulary matters in the design part.

Design matters. Sometimes, it is better to start with a **top-down approach** which consists of implementing the design by mocking underlying dependencies with a `not yet implemented` exception; the input and output of functions must be used though. The code should compile/run while tests will stay red but it is an important step to validate the design first to save further refactoring.

### Example: Top-Down Design in Python

```python
# Step 1: Define the high-level function with proper types
def process_order(order: Order) -> ProcessedOrder:
    """Process an order through validation, pricing, and persistence."""
    # Step 2: Mock dependencies with NotImplementedError
    validated_order = validate_order(order)
    priced_order = calculate_pricing(validated_order)
    saved_order = save_order(priced_order)
    return saved_order

# Step 3: Implement each function one by one
def validate_order(order: Order) -> Order:
    # TODO(agt): Implement order validation
    raise NotImplementedError("Order validation not yet implemented")

def calculate_pricing(order: Order) -> Order:
    # TODO(agt): Implement pricing calculation
    raise NotImplementedError("Pricing calculation not yet implemented")

def save_order(order: Order) -> ProcessedOrder:
    # TODO(agt): Implement order persistence
    raise NotImplementedError("Order persistence not yet implemented")
```

This approach:
- ✅ Validates the design at "compile" time (type checking)
- ✅ Shows the complete workflow
- ✅ Makes it easy to implement piece by piece
- ✅ Keeps tests red until implementation is complete

## Test-Driven Development Workflow

### 1. Red Phase: Write a Failing Test

```python
def test_user_service_register_user():
    # Arrange
    repo = MockUserRepository()
    service = UserService(repo)
    
    # Act
    user = service.register_user("John", "john@example.com")
    
    # Assert
    assert user.name == "John"
    assert user.email == "john@example.com"
    assert repo.save_called
```

### 2. Green Phase: Make it Pass

```python
class UserService:
    def __init__(self, repository: UserRepository):
        self._repository = repository
    
    def register_user(self, name: str, email: str) -> User:
        user = User(
            id=generate_id(),
            name=name,
            email=email
        )
        self._repository.save(user)
        return user
```

### 3. Refactor Phase: Improve the Code

```python
class UserService:
    def __init__(self, repository: UserRepository):
        self._repository = repository
    
    def register_user(self, name: str, email: str) -> User:
        # Extract validation
        self._validate_user_input(name, email)
        
        user = create_user(name, email)
        self._repository.save(user)
        return user
    
    def _validate_user_input(self, name: str, email: str) -> None:
        if not name:
            raise ValidationError("Name is required")
        if not is_valid_email(email):
            raise ValidationError("Invalid email format")
```

## Testing Guidelines

### Pytest Best Practices

```python
import pytest
from myapp.domain.user import User, create_user, ValidationError

# Simple test
def test_create_user_success():
    user = create_user("John", "john@example.com")
    assert user.name == "John"
    assert user.email == "john@example.com"

# Test exception
def test_create_user_invalid_email():
    with pytest.raises(ValidationError):
        create_user("John", "invalid-email")

# Parametrized test (table-driven)
@pytest.mark.parametrize("name,email,should_pass", [
    ("John", "john@example.com", True),
    ("", "john@example.com", False),
    ("John", "", False),
    ("John", "invalid", False),
])
def test_create_user_validation(name: str, email: str, should_pass: bool):
    if should_pass:
        user = create_user(name, email)
        assert user.name == name
    else:
        with pytest.raises(ValidationError):
            create_user(name, email)

# Using fixtures
@pytest.fixture
def user_repository():
    return MockUserRepository()

@pytest.fixture
def user_service(user_repository):
    return UserService(user_repository)

def test_register_user(user_service):
    user = user_service.register_user("John", "john@example.com")
    assert user.name == "John"
```

### Mocking

```python
from unittest.mock import Mock, patch

# Mock object
def test_with_mock():
    mock_repo = Mock(spec=UserRepository)
    mock_repo.find_by_email.return_value = None
    
    service = UserService(mock_repo)
    user = service.register_user("John", "john@example.com")
    
    mock_repo.save.assert_called_once()

# Patch
@patch('myapp.service.user_service.send_email')
def test_registration_sends_email(mock_send_email):
    service = UserService(MockUserRepository())
    service.register_user("John", "john@example.com")
    
    mock_send_email.assert_called_once()
```

### Test Coverage

Aim for **90% code coverage** minimum:

```bash
# Run tests with coverage
pytest tests/ --cov=src --cov-report=html

# View coverage report
open htmlcov/index.html
```

## Git Workflow

### Commit Message Format

Use conventional commits:

```
<type>(agt): <description>

[optional body]

[optional footer]
```

**Types:**
- `feat(agt):` - New feature
- `fix(agt):` - Bug fix
- `refac(agt):` - Refactoring (no functional changes)
- `chore(agt):` - Maintenance tasks
- `doc(agt):` - Documentation
- `test(agt):` - Tests only

**Examples:**
```
feat(agt): add user authentication with JWT

Implement JWT-based authentication for user login.
Includes token generation, validation, and refresh logic.

Closes #123
```

```
fix(agt): resolve email validation bug

Email validation was too strict, rejecting valid emails
with plus signs. Updated regex to allow RFC-compliant emails.
```

```
refac(agt): extract validation logic into separate module

Move all validation functions from domain/user.py to
validation/user_validation.py for better organization.
No functional changes.
```

### Git Rules

**⚠️ CRITICAL:**
- ❌ **NEVER** use `git stash`
- ❌ **NEVER** use `git push` or `git push --force`
- ✅ **ALWAYS** ask user approval before committing

### Commit Workflow

1. **Run tests** and verify coverage
2. **Review changes** with user
3. **Ask for commit approval**
4. **Create commit** with proper message
5. **Confirm** commit was successful

```bash
# Stage changes
git add .

# Commit with message
git commit -m "feat(agt): implement user registration"

# Verify commit
git log -1
```

## Code Review Checklist

Before requesting review:

- [ ] All tests pass
- [ ] Code coverage is at least 90%
- [ ] Type checking passes (mypy)
- [ ] Code is formatted (black)
- [ ] Imports are sorted (isort)
- [ ] No linter warnings (ruff)
- [ ] Type hints are present
- [ ] Docstrings for public APIs
- [ ] Documentation is updated
- [ ] Commit message follows format

## Best Practices Summary

1. **TDD**: Write tests first (Red-Green-Refactor)
2. **DDD**: Separate business and technical concerns
3. **Top-Down**: Design with mocks, implement incrementally
4. **Type Hints**: Always use type annotations
5. **Dataclasses**: For data containers
6. **Testing**: pytest with parametrize, 90% coverage
7. **Git**: Conventional commits, user approval required

## Resources

- [AGENTS_RULES.md](./AGENTS_RULES.md) - Complete AI agent rules
- [CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md) - Quality principles
- [Python Docs](https://docs.python.org/3/)
- [PEP 8](https://peps.python.org/pep-0008/)
- [pytest](https://docs.pytest.org/)

---

**Remember**: Read [AGENTS_RULES.md](./AGENTS_RULES.md) completely before starting any work.
