# Code Quality Principles for Python

**⚠️ CRITICAL: Read these principles before code reviews and refactoring**

## 1. Avoid Bad Trade-offs with Default Values

### ❌ **Problem**: Creating default/fallback values to mask missing data

```python
# BAD: Hiding potential issues with defaults
def get_rate_plan(room_stay: RoomStay) -> RatePlan:
    if not room_stay.rate_plans:
        # Creating fake data with many defaults
        return RatePlan(
            code="",
            commission=0,
            cancel_deadline=None,
            is_non_refundable=False,
            # ... many more defaults
        )
    return room_stay.rate_plans[0]
```

### ✅ **Solution**: Fail fast with proper error handling

```python
# GOOD: Explicit error handling, no hidden defaults
def get_rate_plan(room_stay: RoomStay) -> RatePlan:
    if not room_stay.rate_plans:
        raise ValueError("No rate plans available")
    return room_stay.rate_plans[0]

# Usage
try:
    rate_plan = get_rate_plan(room_stay)
except ValueError as e:
    logger.error(f"Failed to get rate plan: {e}")
    raise
```

### **Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ❌ Hides problems until production
- ✅ Fail-fast reveals issues immediately
- ✅ Proper exceptions enable better handling
- ✅ Consistent error propagation

## 2. Type Hints Everywhere

### ❌ **Problem**: No type information

```python
# BAD: No type hints
def process_users(users):
    return [u for u in users if u.age > 18]

def calculate_total(items):
    return sum(item.price for item in items)
```

### ✅ **Solution**: Always use type hints

```python
# GOOD: Clear types
from typing import List

def process_users(users: List[User]) -> List[User]:
    return [u for u in users if u.age > 18]

def calculate_total(items: List[Item]) -> float:
    return sum(item.price for item in items)
```

### **Why it matters**:
- ✅ Catches errors at development time (mypy)
- ✅ Better IDE support (autocomplete, refactoring)
- ✅ Self-documenting code
- ✅ Easier to maintain and understand

## 3. Use Dataclasses for Data Containers

### ❌ **Problem**: Using dicts or tuples for structured data

```python
# BAD: Dict for structured data
user = {
    "name": "John",
    "email": "john@example.com",
    "age": 30
}

# BAD: Tuple with positional access
user = ("John", "john@example.com", 30)
name = user[0]  # What is index 0?
```

### ✅ **Solution**: Use dataclasses

```python
# GOOD: Dataclass with types
from dataclasses import dataclass
from typing import Optional

@dataclass(frozen=True)
class User:
    name: str
    email: str
    age: Optional[int] = None

user = User(name="John", email="john@example.com", age=30)
print(user.name)  # Clear and type-safe
```

### **Why it matters**:
- ✅ Type safety
- ✅ Immutability (with frozen=True)
- ✅ Clear field names
- ✅ Better IDE support

## 4. Maintain Consistency Across Similar Code Paths

Same problem = same solution. When you encounter similar scenarios in the codebase, use the same approach to handle them.

### ❌ **Problem**: Inconsistent error handling

```python
# BAD: Mixing error handling strategies
def process_order(order: Order) -> ProcessResult | None:
    if not order.is_valid():
        return None  # Sometimes returns None
    
    if not order.has_items():
        raise ValueError("No items")  # Sometimes raises
    
    return ProcessResult(success=True)
```

### ✅ **Solution**: Consistent approach

```python
# GOOD: Consistent error handling
def process_order(order: Order) -> ProcessResult:
    if not order.is_valid():
        raise ValidationError("Invalid order")
    
    if not order.has_items():
        raise ValidationError("No items in order")
    
    return ProcessResult(success=True)
```

### Why it matters:
- ✅ Makes code predictable and easier to understand
- ✅ Reduces cognitive load when reading code
- ✅ Makes refactoring safer
- ✅ Helps identify patterns and abstractions

## 5. Extract Reusable Functions

Organize code for reuse. When you see repeated logic, extract it into a well-named function.

### ❌ **Problem**: Repeated validation logic

```python
# BAD: Duplicated validation
def create_user(name: str, email: str) -> User:
    if not name:
        raise ValueError("Name is required")
    if "@" not in email:
        raise ValueError("Invalid email")
    # ...

def update_user(user: User, email: str) -> User:
    # Same validation duplicated
    if "@" not in email:
        raise ValueError("Invalid email")
    # ...
```

### ✅ **Solution**: Extract to reusable function

```python
# GOOD: Extracted validation
def validate_name(name: str) -> None:
    if not name:
        raise ValueError("Name is required")

def validate_email(email: str) -> None:
    if "@" not in email:
        raise ValueError("Invalid email format")

def create_user(name: str, email: str) -> User:
    validate_name(name)
    validate_email(email)
    # ...

def update_user(user: User, email: str) -> User:
    validate_email(email)
    # ...
```

### Why it matters:
- ✅ Reduces code duplication
- ✅ Makes changes easier (single point of modification)
- ✅ Improves testability
- ✅ Creates a vocabulary of reusable operations

## 6. Python-Specific Quality Principles

### Use List Comprehensions (When Readable)

```python
# GOOD: Readable comprehension
active_users = [u for u in users if u.is_active]

# BAD: Too complex
result = [x for sublist in [[y for y in range(10) if y % 2] for _ in range(5)] for x in sublist]

# BETTER: Use functions for complex logic
def get_even_numbers(n: int) -> List[int]:
    return [y for y in range(n) if y % 2 == 0]

result = [x for _ in range(5) for x in get_even_numbers(10)]
```

### Use Context Managers for Resources

```python
# GOOD: Context manager ensures cleanup
with open("file.txt") as f:
    data = f.read()

# BAD: Manual cleanup (error-prone)
f = open("file.txt")
try:
    data = f.read()
finally:
    f.close()
```

### Use f-strings for Formatting

```python
# GOOD: f-strings (readable, fast)
message = f"User {user.name} has {user.age} years"

# BAD: Old-style formatting
message = "User %s has %d years" % (user.name, user.age)
message = "User {} has {} years".format(user.name, user.age)
```

## Summary

1. **Fail Fast**: No default values to hide errors
2. **Type Hints**: Always use type annotations
3. **Dataclasses**: For data containers
4. **Be Consistent**: Same problem = same solution
5. **Extract & Reuse**: DRY principle
6. **Comprehensions**: When readable
7. **Context Managers**: For resources
8. **f-strings**: For formatting

**Remember**: These principles make code more maintainable, testable, and reliable.
