# AI Agent Rules for Python Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on Python projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
pytest tests/ -v --cov=src --cov-report=html --cov-report=term
mypy src/
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check coverage report for new/modified code (minimum 90% coverage)
3. **Type checking passes** - mypy should report no errors

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Domain-Driven Design & Test Types

**Acceptance Tests (Domain-Focused)**
- **Focus**: Business rules, domain input/output
- **Purpose**: Test business behavior, not implementation details
- **Robustness**: Should survive refactoring (high-level, stable contracts)
- **Vocabulary**: Use business/domain terminology
- **When**: Test features from user/business perspective

**Unit Tests (Technical-Focused)**
- **Focus**: Simple, isolated components (business OR technical)
- **Purpose**: Test individual functions, algorithms, patterns
- **Scope**: Lower-level than acceptance tests

#### Separation of Concerns
- **Business rules** vs **Technical rules** must be clearly separated
- Domain-Driven Design: Start with business problems and vocabulary first

#### Test Organization
- Use pytest fixtures for setup/teardown
- Use parametrize for table-driven tests
- Use mocks/patches for external dependencies
- Organize tests in test_*.py files

---

## Development

### Python-Specific Best Practices

#### 1. Type Hints (Python 3.10+)
- **Always use type hints** for function signatures
- **Use from __future__ import annotations** for forward references
- **Use typing module** for complex types
- **Run mypy** to verify type correctness

```python
from typing import Optional, List, Dict
from dataclasses import dataclass

@dataclass
class User:
    name: str
    email: str
    age: Optional[int] = None

def create_user(name: str, email: str) -> User:
    if not name or not email:
        raise ValueError("Name and email are required")
    return User(name=name, email=email)

def find_users_by_age(users: List[User], min_age: int) -> List[User]:
    return [u for u in users if u.age and u.age >= min_age]
```

#### 2. Dataclasses
- **Use dataclasses** for data containers
- **Use frozen=True** for immutable data
- **Use slots=True** (Python 3.10+) for memory efficiency

```python
from dataclasses import dataclass, field
from typing import List

@dataclass(frozen=True)
class Order:
    id: str
    items: List[str] = field(default_factory=list)
    total: float = 0.0
```

#### 3. Error Handling
- **Use exceptions** for exceptional cases
- **Create custom exceptions** for domain errors
- **Use Result pattern** for expected failures
- **Always provide context** in error messages

```python
class ValidationError(Exception):
    """Raised when validation fails."""
    pass

def validate_email(email: str) -> None:
    if "@" not in email:
        raise ValidationError(f"Invalid email format: {email}")
```

#### 4. Result Pattern (Optional)
```python
from typing import Generic, TypeVar, Union
from dataclasses import dataclass

T = TypeVar('T')
E = TypeVar('E')

@dataclass
class Ok(Generic[T]):
    value: T
    
    def is_ok(self) -> bool:
        return True
    
    def is_err(self) -> bool:
        return False

@dataclass
class Err(Generic[E]):
    error: E
    
    def is_ok(self) -> bool:
        return False
    
    def is_err(self) -> bool:
        return True

Result = Union[Ok[T], Err[E]]

def divide(x: float, y: float) -> Result[float, str]:
    if y == 0:
        return Err("division by zero")
    return Ok(x / y)

# Usage
result = divide(10, 2)
if result.is_ok():
    print(f"Result: {result.value}")
else:
    print(f"Error: {result.error}")
```

#### 5. Context Managers
- **Use with statement** for resource management
- **Create custom context managers** when needed
- **Use contextlib** for simple cases

```python
from contextlib import contextmanager
from typing import Iterator

@contextmanager
def database_transaction(conn: Connection) -> Iterator[Connection]:
    try:
        yield conn
        conn.commit()
    except Exception:
        conn.rollback()
        raise
    finally:
        conn.close()

# Usage
with database_transaction(get_connection()) as conn:
    conn.execute("INSERT ...")
```

#### 6. List/Dict Comprehensions
- **Use comprehensions** for transformations
- **Keep them readable** - use loops if too complex
- **Use generator expressions** for large datasets

```python
# Good: Readable comprehension
active_users = [u for u in users if u.is_active]

# Good: Dict comprehension
user_map = {u.id: u.name for u in users}

# Bad: Too complex
result = [x for sublist in [[y for y in range(10) if y % 2] for _ in range(5)] for x in sublist]

# Better: Use functions
def get_even_numbers(n: int) -> List[int]:
    return [y for y in range(n) if y % 2 == 0]

result = [x for _ in range(5) for x in get_even_numbers(10)]
```

#### 7. Decorators
- **Use decorators** for cross-cutting concerns
- **Use functools.wraps** to preserve metadata
- **Keep decorators simple**

```python
from functools import wraps
from typing import Callable, TypeVar

T = TypeVar('T')

def log_calls(func: Callable[..., T]) -> Callable[..., T]:
    @wraps(func)
    def wrapper(*args, **kwargs) -> T:
        print(f"Calling {func.__name__}")
        result = func(*args, **kwargs)
        print(f"Finished {func.__name__}")
        return result
    return wrapper

@log_calls
def process_data(data: List[int]) -> int:
    return sum(data)
```

### Code Quality Principles

#### 1. Avoid Default Values - Fail Fast
```python
# Bad: Default value hides error
def get_config(key: str) -> str:
    try:
        return config[key]
    except KeyError:
        return "default"  # Silently fails

# Good: Explicit error handling
def get_config(key: str) -> str:
    if key not in config:
        raise KeyError(f"Config key '{key}' not found")
    return config[key]
```

#### 2. Type Hints Everywhere
```python
# Bad: No type hints
def process(data):
    return [x * 2 for x in data]

# Good: Clear types
def process(data: List[int]) -> List[int]:
    return [x * 2 for x in data]
```

#### 3. Use Dataclasses for Data
```python
# Bad: Dict for structured data
user = {"name": "John", "email": "john@example.com"}

# Good: Dataclass with types
@dataclass
class User:
    name: str
    email: str

user = User(name="John", email="john@example.com")
```

---

## Commit

### Git Workflow Rules

**⚠️ CRITICAL: Git Rules**

1. **NEVER use `git stash`** - Always commit or discard changes explicitly
2. **NEVER use `git push`** - User will push manually
3. **NEVER use `git push --force`** - Absolutely forbidden
4. **ALWAYS ask user approval** before committing

### Commit Message Format

Use conventional commits format:

```
<type>(agt): <description>

[optional body]
```

**Types:**
- `feat(agt):` - New feature
- `fix(agt):` - Bug fix
- `refac(agt):` - Refactoring (no functional changes)
- `chore(agt):` - Maintenance tasks
- `doc(agt):` - Documentation
- `test(agt):` - Tests only

**Examples:**
```
feat(agt): add user authentication with JWT
fix(agt): resolve race condition in order processing
refac(agt): extract validation logic into separate module
test(agt): add parametrized tests for email validation
```

---

## Python-Specific Tools

### Testing
```bash
# Run all tests
pytest tests/

# Run with coverage
pytest tests/ --cov=src --cov-report=html

# Run specific test
pytest tests/test_user.py::test_create_user

# Run with markers
pytest -m "not slow"

# Watch mode (with pytest-watch)
ptw
```

### Type Checking
```bash
# Check types
mypy src/

# Strict mode
mypy --strict src/

# Specific file
mypy src/domain/user.py
```

### Code Formatting
```bash
# Format code
black src/ tests/

# Check formatting
black --check src/ tests/

# Sort imports
isort src/ tests/
```

### Linting
```bash
# Lint with ruff (fast)
ruff check src/ tests/

# Fix auto-fixable issues
ruff check --fix src/ tests/

# Lint with pylint
pylint src/
```

### Dependencies
```bash
# Install project
pip install -e .

# Install dev dependencies
pip install -e ".[dev]"

# Update dependencies
pip install --upgrade -r requirements.txt

# Freeze dependencies
pip freeze > requirements.txt
```

---

## Python Project Structure

### Standard Layout

```
myproject/
├── src/
│   └── myapp/
│       ├── __init__.py
│       ├── domain/
│       │   ├── __init__.py
│       │   ├── user.py
│       │   └── order.py
│       ├── repository/
│       │   ├── __init__.py
│       │   └── user_repository.py
│       ├── service/
│       │   ├── __init__.py
│       │   └── user_service.py
│       └── api/
│           ├── __init__.py
│           └── routes.py
├── tests/
│   ├── __init__.py
│   ├── test_user.py
│   └── test_user_service.py
├── docs/
├── pyproject.toml
├── setup.py or setup.cfg
├── requirements.txt
├── requirements-dev.txt
├── .gitignore
├── README.md
└── mypy.ini
```

---

## Additional Resources

### Official Documentation
- [Python Documentation](https://docs.python.org/3/)
- [PEP 8 Style Guide](https://peps.python.org/pep-0008/)
- [Type Hints (PEP 484)](https://peps.python.org/pep-0484/)

### Testing
- [pytest Documentation](https://docs.pytest.org/)
- [pytest-cov](https://pytest-cov.readthedocs.io/)
- [unittest.mock](https://docs.python.org/3/library/unittest.mock.html)

### Type Checking
- [mypy Documentation](https://mypy.readthedocs.io/)
- [typing Module](https://docs.python.org/3/library/typing.html)

### Tools
- [Black](https://black.readthedocs.io/) - Code formatter
- [Ruff](https://docs.astral.sh/ruff/) - Fast linter
- [isort](https://pycqa.github.io/isort/) - Import sorter

---

**Remember**: This file is your primary reference. Read it completely before starting any work on a Python project.
