# Python AI Agents - Quick Reference

**📖 For complete documentation, see [AGENTS_RULES.md](./AGENTS_RULES.md)**

## Essential Workflow

```
Plan → Test → Develop → Test → Commit
```

1. **Plan**: Discuss with user, create todo-list, get approval
2. **Test**: Write failing tests first (TDD)
3. **Develop**: Implement with Python best practices
4. **Test**: Run `pytest --cov` (90% minimum)
5. **Commit**: Get user approval, use conventional format

## Python Best Practices

### Type Hints
```python
from typing import Optional, List

def create_user(name: str, email: str) -> User:
    return User(name=name, email=email)

def find_users(min_age: int) -> List[User]:
    return [u for u in users if u.age and u.age >= min_age]
```

### Dataclasses
```python
from dataclasses import dataclass

@dataclass(frozen=True)
class User:
    name: str
    email: str
    age: Optional[int] = None
```

### Error Handling
```python
# Use exceptions for exceptional cases
def validate_email(email: str) -> None:
    if "@" not in email:
        raise ValidationError(f"Invalid email: {email}")

# Or use Result pattern
def divide(x: float, y: float) -> Result[float, str]:
    if y == 0:
        return Err("division by zero")
    return Ok(x / y)
```

### Context Managers
```python
with open("file.txt") as f:
    data = f.read()

# Custom context manager
@contextmanager
def transaction(conn: Connection) -> Iterator[Connection]:
    try:
        yield conn
        conn.commit()
    except:
        conn.rollback()
        raise
```

## Testing

### Pytest
```python
def test_create_user():
    user = create_user("John", "john@example.com")
    assert user.name == "John"
    assert user.email == "john@example.com"

# Parametrize for table-driven tests
@pytest.mark.parametrize("email,valid", [
    ("test@example.com", True),
    ("invalid", False),
    ("", False),
])
def test_validate_email(email: str, valid: bool):
    if valid:
        validate_email(email)  # Should not raise
    else:
        with pytest.raises(ValidationError):
            validate_email(email)
```

### Run Tests
```bash
# All tests with coverage
pytest tests/ -v --cov=src --cov-report=html

# Type check
mypy src/

# Format
black src/ tests/

# Lint
ruff check src/ tests/
```

## Code Quality

### ❌ Don't: Hide Errors
```python
def get_config(key: str) -> str:
    try:
        return config[key]
    except KeyError:
        return "default"  # BAD: Silent failure
```

### ✅ Do: Explicit Errors
```python
def get_config(key: str) -> str:
    if key not in config:
        raise KeyError(f"Config key '{key}' not found")
    return config[key]
```

## Git Workflow

### Rules
- ❌ **NEVER** `git stash`
- ❌ **NEVER** `git push` or `--force`
- ✅ **ALWAYS** ask user approval before commit

### Commit Format
```
<type>(agt): <description>

feat(agt): add user registration
fix(agt): resolve validation bug
refac(agt): extract email validation
test(agt): add parametrized tests
```

## Common Commands

```bash
# Testing
pytest tests/                    # Run all tests
pytest tests/ --cov=src          # With coverage
pytest -k test_user              # Specific test

# Type checking
mypy src/                        # Check types
mypy --strict src/               # Strict mode

# Formatting
black src/ tests/                # Format code
isort src/ tests/                # Sort imports

# Linting
ruff check src/ tests/           # Fast lint
ruff check --fix src/            # Auto-fix

# Dependencies
pip install -e .                 # Install project
pip install -e ".[dev]"          # With dev deps
```

## Project Structure

```
myproject/
├── src/
│   └── myapp/
│       ├── domain/
│       ├── repository/
│       └── service/
├── tests/
├── pyproject.toml
└── requirements.txt
```

## Quick Tips

1. **Type hints**: Always use them
2. **Dataclasses**: For data containers
3. **f-strings**: For string formatting
4. **Comprehensions**: When readable
5. **Context managers**: For resources
6. **pytest**: For testing
7. **mypy**: For type checking
8. **black**: For formatting

## Resources

- [AGENTS_RULES.md](./AGENTS_RULES.md) - Complete rules
- [CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md) - Quality guidelines
- [CONTRIBUTING.md](./CONTRIBUTING.md) - Contribution guide
- [Python Docs](https://docs.python.org/3/)
- [PEP 8](https://peps.python.org/pep-0008/)
