# PHP Project Template

This directory contains a complete, autonomous AI Agent configuration for PHP projects.

## For AI Agents

**Start here**: Read the [AGENTS_RULES.md](./AGENTS_RULES.md) file for complete rules.

All necessary guidelines are in this directory:
- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete rules (works with all AI assistants)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE (auto-loaded)
- **[AGENTS.md](./AGENTS.md)** - Quick reference
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guide

## For Developers

When starting a new PHP project in this directory:

1. Follow the Quick Start guide below
2. Ensure the AI agent reads the `AGENTS_RULES.md` file
3. Follow TDD and Domain-Driven Design principles

## Structure

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (universal)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines
- **[README.md](./README.md)** - This file

## Key Requirements

- **Test-Driven Development**: Write tests first
- **90% Code Coverage**: Minimum requirement
- **Domain-Driven Design**: Business vocabulary first
- **Result Pattern**: Use Result types for error handling
- **Strict Types**: `declare(strict_types=1)` in every file
- **Readonly Properties**: For immutable value objects
- **Type Safety**: Wrap primitives, use enums
- **Modern PHP**: Use PHP 8.1+ features

## Getting Started

### Prerequisites

- PHP 8.1 or higher (8.3+ recommended)
- Composer

### Initialize Project

```bash
# Create composer.json
composer init

# Install dependencies
composer require --dev phpunit/phpunit
composer require --dev phpstan/phpstan
composer require --dev friendsofphp/php-cs-fixer

# Create directory structure
mkdir -p src tests/Unit tests/Integration
```

### composer.json Example

```json
{
    "name": "vendor/my-project",
    "description": "My PHP Project",
    "type": "project",
    "require": {
        "php": "^8.3"
    },
    "require-dev": {
        "phpunit/phpunit": "^10.5",
        "phpstan/phpstan": "^1.10",
        "friendsofphp/php-cs-fixer": "^3.40"
    },
    "autoload": {
        "psr-4": {
            "App\\": "src/"
        }
    },
    "autoload-dev": {
        "psr-4": {
            "Tests\\": "tests/"
        }
    },
    "config": {
        "optimize-autoloader": true,
        "sort-packages": true
    }
}
```

### phpunit.xml Example

```xml
<?xml version="1.0" encoding="UTF-8"?>
<phpunit xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
         xsi:noNamespaceSchemaLocation="vendor/phpunit/phpunit/phpunit.xsd"
         bootstrap="vendor/autoload.php"
         colors="true">
    <testsuites>
        <testsuite name="Unit">
            <directory>tests/Unit</directory>
        </testsuite>
        <testsuite name="Integration">
            <directory>tests/Integration</directory>
        </testsuite>
    </testsuites>
    <coverage>
        <include>
            <directory suffix=".php">src</directory>
        </include>
        <report>
            <html outputDirectory="coverage"/>
        </report>
    </coverage>
</phpunit>
```

## Example Project Structure

```
my-project/
├── src/
│   ├── Common/
│   │   ├── Result.php
│   │   ├── Success.php
│   │   └── Failure.php
│   ├── Domain/
│   │   ├── Order.php
│   │   ├── Customer.php
│   │   ├── OrderStatus.php (enum)
│   │   └── ValueObjects/
│   │       ├── OrderId.php
│   │       └── Money.php
│   ├── Service/
│   │   ├── OrderService.php
│   │   └── PaymentService.php
│   └── Repository/
│       └── OrderRepository.php (interface)
├── tests/
│   ├── Unit/
│   │   ├── Domain/
│   │   │   └── OrderTest.php
│   │   └── Service/
│   │       └── OrderServiceTest.php
│   └── Integration/
│       └── OrderWorkflowTest.php
├── composer.json
├── phpunit.xml
└── README.md
```

## Example Code

### Value Object (Readonly)

```php
<?php

declare(strict_types=1);

namespace App\Domain\ValueObjects;

final readonly class OrderId
{
    public function __construct(public string $value)
    {
        if (empty($value)) {
            throw new \InvalidArgumentException('OrderId cannot be empty');
        }
    }
}

final readonly class Money
{
    public function __construct(
        public float $amount,
        public string $currency
    ) {
        if ($amount < 0) {
            throw new \InvalidArgumentException('Amount cannot be negative');
        }
    }

    public static function usd(float $amount): self
    {
        return new self($amount, 'USD');
    }
}
```

### Enum (PHP 8.1+)

```php
<?php

declare(strict_types=1);

namespace App\Domain;

enum OrderStatus: string
{
    case PENDING = 'pending';
    case CONFIRMED = 'confirmed';
    case SHIPPED = 'shipped';
    case DELIVERED = 'delivered';

    public function getLabel(): string
    {
        return match ($this) {
            self::PENDING => 'Order is pending',
            self::CONFIRMED => 'Order confirmed',
            self::SHIPPED => 'Shipped',
            self::DELIVERED => 'Delivered',
        };
    }
}
```

### Domain Entity

```php
<?php

declare(strict_types=1);

namespace App\Domain;

use App\Domain\ValueObjects\OrderId;
use App\Domain\ValueObjects\Money;

final readonly class Order
{
    /**
     * @param OrderId $id
     * @param Customer $customer
     * @param array<OrderItem> $items
     * @param OrderStatus $status
     */
    public function __construct(
        public OrderId $id,
        public Customer $customer,
        public array $items,
        public OrderStatus $status
    ) {
        if (empty($items)) {
            throw new \InvalidArgumentException('Order must have at least one item');
        }
    }

    public function calculateTotal(): Money
    {
        $total = array_reduce(
            $this->items,
            fn(float $sum, OrderItem $item) => $sum + $item->total->amount,
            0.0
        );
        return Money::usd($total);
    }
}
```

### Result Pattern

```php
<?php

declare(strict_types=1);

namespace App\Common;

/**
 * @template T
 */
abstract class Result
{
    /**
     * @template U
     * @param U $value
     * @return Success<U>
     */
    public static function success(mixed $value): Success
    {
        return new Success($value);
    }

    /**
     * @template U
     * @param string $error
     * @return Failure<U>
     */
    public static function failure(string $error): Failure
    {
        return new Failure($error);
    }

    abstract public function map(callable $mapper): Result;
    abstract public function flatMap(callable $mapper): Result;
    abstract public function isSuccess(): bool;
    abstract public function isFailure(): bool;
}
```

### Service with Result

```php
<?php

declare(strict_types=1);

namespace App\Service;

use App\Common\Result;
use App\Domain\Order;

final class OrderService
{
    public function __construct(
        private readonly OrderRepository $orderRepository,
        private readonly PaymentProcessor $paymentProcessor
    ) {
    }

    public function validateOrder(Order $order): Result
    {
        if (empty($order->items)) {
            return Result::failure('Order has no items');
        }

        if ($order->calculateTotal()->amount <= 0) {
            return Result::failure('Order total must be positive');
        }

        return Result::success($order);
    }

    public function processOrder(Order $order): Result
    {
        return $this->validateOrder($order)
            ->flatMap(fn($o) => $this->paymentProcessor->process($o))
            ->flatMap(fn($o) => $this->orderRepository->save($o));
    }
}
```

### Test Example (PHPUnit)

```php
<?php

declare(strict_types=1);

namespace Tests\Unit\Service;

use PHPUnit\Framework\TestCase;
use App\Service\OrderService;
use App\Domain\Order;
use App\Domain\OrderStatus;
use App\Domain\ValueObjects\OrderId;
use App\Domain\ValueObjects\Money;

final class OrderServiceTest extends TestCase
{
    public function testShouldValidateOrderSuccessfully(): void
    {
        // Given
        $order = new Order(
            id: new OrderId('ORD-001'),
            customer: new Customer('John Doe'),
            items: [new OrderItem('item1', Money::usd(10.0))],
            status: OrderStatus::PENDING
        );
        $service = new OrderService();

        // When
        $result = $service->validateOrder($order);

        // Then
        $this->assertTrue($result->isSuccess());
        $this->assertEquals($order, $result->value);
    }

    public function testShouldFailWhenOrderHasNoItems(): void
    {
        // Given
        $order = new Order(
            id: new OrderId('ORD-001'),
            customer: new Customer('John Doe'),
            items: [],
            status: OrderStatus::PENDING
        );
        $service = new OrderService();

        // When
        $result = $service->validateOrder($order);

        // Then
        $this->assertTrue($result->isFailure());
        $this->assertEquals('Order has no items', $result->error);
    }
}
```

## Running Tests

```bash
# Run all tests
./vendor/bin/phpunit

# Run specific test
./vendor/bin/phpunit tests/Unit/Service/OrderServiceTest.php

# Run with coverage
./vendor/bin/phpunit --coverage-html coverage/
./vendor/bin/phpunit --coverage-text

# View coverage report
# Open coverage/index.html in browser
```

## Code Quality

```bash
# PHPStan - Static analysis
./vendor/bin/phpstan analyse src tests

# PHP CS Fixer - Code style
./vendor/bin/php-cs-fixer fix

# Or check without fixing
./vendor/bin/php-cs-fixer fix --dry-run --diff
```

---

## Complete Documentation

All documentation is self-contained in this directory:

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (universal)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines
- **[README.md](./README.md)** - This file

**Note**: This directory is completely autonomous and works with any AI assistant (Cursor, GitHub Copilot, ChatGPT, Claude, etc.).
