# AI Agent Rules for PHP Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on PHP projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
# PHPUnit
./vendor/bin/phpunit

# With coverage
./vendor/bin/phpunit --coverage-html coverage/
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check coverage reports for new/modified code (minimum 90% coverage)

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Domain-Driven Design & Test Types

**Acceptance Tests (Domain-Focused)**
- **Focus**: Business rules, domain input/output
- **Purpose**: Test business behavior, not implementation details
- **Robustness**: Should survive refactoring (high-level, stable contracts)
- **Vocabulary**: Use business/domain terminology
- **When**: Test features from user/business perspective

**Unit Tests (Technical-Focused)**
- **Focus**: Simple, isolated components (business OR technical)
- **Purpose**: Test individual methods, algorithms, patterns
- **Scope**: Lower-level than acceptance tests

#### Separation of Concerns
- **Business rules** vs **Technical rules** must be clearly separated
- Domain-Driven Design: Start with business problems and vocabulary first
- Technical code (algorithms, patterns) implements the domain model

#### The "Red" Phase Matters
1. **Design First**: The "red" phase validates design at compile time
2. **Top-Down Approach**: When appropriate, implement design by:
   - Defining method signatures with correct input/output types
   - Using business vocabulary in types and method names
   - Mocking dependencies with `throw new \LogicException("Not implemented yet")`
   - **Code must compile** even with mocks
   - Tests stay red but design is validated
3. **Design Quality Indicator**: If underlying parts are easy to implement, the design is good

#### Make It Work, Then Make It Right
- When implementing features that require refactoring:
  1. Make it work (get tests passing)
  2. Make it right (refactor without changing acceptance tests)
- **Acceptance tests should NOT change during refactoring**
- This proves the refactoring preserves business behavior

### Test Commands

```bash
# Composer
composer install

# PHPUnit
./vendor/bin/phpunit
./vendor/bin/phpunit tests/Unit/OrderServiceTest.php
./vendor/bin/phpunit --coverage-html coverage/
./vendor/bin/phpunit --coverage-text

# Pest (alternative)
./vendor/bin/pest
./vendor/bin/pest --coverage
```

---

## Development

### Build Environment

```bash
# Composer
composer install
composer update
composer dump-autoload

# Development server
php -S localhost:8000 -t public/

# Laravel Artisan
php artisan serve
php artisan test
```

### Code Quality Principles

#### 1. Avoid Bad Trade-offs with Default Values

**❌ Problem**: Creating default/fallback values to mask missing data

```php
// BAD: Hiding potential issues with defaults
$minimalRatePlan = $roomStayPrice->getRatePlans()[0] ?? new RatePlan(
    ratePlanCode: null,
    commission: new Commission(0, CommissionStatus::FULL),
    cancelPenalty: new CancelPenalty(null, false, null),
    // ... many more defaults
);
```

**✅ Solution**: Fail fast with proper error handling

```php
// GOOD: Explicit error handling, no hidden defaults
$ratePlanResult = !empty($roomStayPrice->getRatePlans())
    ? Result::success($roomStayPrice->getRatePlans()[0])
    : Result::failure('No rate plan found');
```

```php
// GOOD: Explicit failure to mock and decide later
$guaranteeForCheck = throw new \LogicException(
    'TODO(agt): Properly convert GuaranteeTerm to Guarantee'
);

// BAD: Hides with a default value trade-offs
// TODO(agt): Properly convert GuaranteeTerm to Guarantee
$guaranteeForCheck = null;
```

**Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling

#### 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution

#### 3. Extract Reusable Functions to Modules

Organize for reuse

### TODO Comments

When adding todo in the code base, use this prefix: `//TODO(agt)`

---

## PHP Specific Guidelines

### PHP Version
- **Minimum**: PHP 8.1
- **Recommended**: PHP 8.3+
- Use modern PHP features (readonly properties, enums, attributes, named arguments, etc.)

### Required Patterns

#### Strict Types (Always)
✅ **Always declare strict types at the top of every file**

```php
<?php

declare(strict_types=1);

namespace App\Domain;

// Your code here
```

#### Result Pattern (Railway-Oriented Programming)
✅ **Use Result pattern for error handling**

```php
<?php

declare(strict_types=1);

namespace App\Common;

/**
 * @template T
 */
abstract class Result
{
    /**
     * @template U
     * @param U $value
     * @return Success<U>
     */
    public static function success(mixed $value): Success
    {
        return new Success($value);
    }

    /**
     * @template U
     * @param string $error
     * @return Failure<U>
     */
    public static function failure(string $error): Failure
    {
        return new Failure($error);
    }

    /**
     * @template U
     * @param callable(T): U $mapper
     * @return Result<U>
     */
    abstract public function map(callable $mapper): Result;

    /**
     * @template U
     * @param callable(T): Result<U> $mapper
     * @return Result<U>
     */
    abstract public function flatMap(callable $mapper): Result;

    abstract public function isSuccess(): bool;
    abstract public function isFailure(): bool;
}

/**
 * @template T
 * @extends Result<T>
 */
final class Success extends Result
{
    /**
     * @param T $value
     */
    public function __construct(public readonly mixed $value)
    {
    }

    public function map(callable $mapper): Result
    {
        return Result::success($mapper($this->value));
    }

    public function flatMap(callable $mapper): Result
    {
        return $mapper($this->value);
    }

    public function isSuccess(): bool
    {
        return true;
    }

    public function isFailure(): bool
    {
        return false;
    }
}

/**
 * @template T
 * @extends Result<T>
 */
final class Failure extends Result
{
    public function __construct(public readonly string $error)
    {
    }

    public function map(callable $mapper): Result
    {
        return $this;
    }

    public function flatMap(callable $mapper): Result
    {
        return $this;
    }

    public function isSuccess(): bool
    {
        return false;
    }

    public function isFailure(): bool
    {
        return true;
    }
}

// Usage
function validateOrder(Order $order): Result
{
    return validateCustomer($order)
        ->flatMap(fn($o) => validateItems($o))
        ->flatMap(fn($o) => calculateTotal($o));
}
```

#### Readonly Properties (PHP 8.1+)
✅ **Use readonly for immutable value objects**

```php
<?php

declare(strict_types=1);

namespace App\Domain\ValueObjects;

final readonly class OrderId
{
    public function __construct(public string $value)
    {
        if (empty($value)) {
            throw new \InvalidArgumentException('OrderId cannot be empty');
        }
    }
}

final readonly class Money
{
    public function __construct(
        public float $amount,
        public string $currency
    ) {
        if ($amount < 0) {
            throw new \InvalidArgumentException('Amount cannot be negative');
        }
    }

    public static function usd(float $amount): self
    {
        return new self($amount, 'USD');
    }
}
```

#### Enums (PHP 8.1+)
✅ **Use enums for closed sets of values**

```php
<?php

declare(strict_types=1);

namespace App\Domain;

enum OrderStatus: string
{
    case PENDING = 'pending';
    case CONFIRMED = 'confirmed';
    case SHIPPED = 'shipped';
    case DELIVERED = 'delivered';

    public function getLabel(): string
    {
        return match ($this) {
            self::PENDING => 'Order is pending',
            self::CONFIRMED => 'Order confirmed',
            self::SHIPPED => 'Shipped',
            self::DELIVERED => 'Delivered',
        };
    }
}

enum CommissionStatus: string
{
    case FULL = 'full';
    case PARTIAL = 'partial';
    case NONE = 'none';
}
```

#### Named Arguments
✅ **Use named arguments for clarity**

```php
<?php

declare(strict_types=1);

// GOOD - Clear and explicit
$order = new Order(
    id: new OrderId('ORD-001'),
    customer: $customer,
    items: $items,
    status: OrderStatus::PENDING
);

// Also good for optional parameters
$result = processOrder(
    order: $order,
    validateInventory: true,
    sendNotification: false
);
```

#### Constructor Property Promotion (PHP 8.0+)
✅ **Use constructor property promotion**

```php
<?php

declare(strict_types=1);

namespace App\Domain;

final readonly class Order
{
    public function __construct(
        public OrderId $id,
        public Customer $customer,
        public array $items,
        public OrderStatus $status
    ) {
        if (empty($items)) {
            throw new \InvalidArgumentException('Order must have at least one item');
        }
    }

    public function calculateTotal(): Money
    {
        $total = array_reduce(
            $this->items,
            fn($sum, $item) => $sum + $item->total->amount,
            0.0
        );
        return Money::usd($total);
    }
}
```

#### Dependency Injection
✅ **Use constructor injection**

```php
<?php

declare(strict_types=1);

namespace App\Service;

final class OrderService
{
    public function __construct(
        private readonly OrderRepository $orderRepository,
        private readonly PaymentProcessor $paymentProcessor,
        private readonly LoggerInterface $logger
    ) {
    }

    public function processOrder(Order $order): Result
    {
        $this->logger->info('Processing order', ['orderId' => $order->id->value]);

        return $this->validateOrder($order)
            ->flatMap(fn($o) => $this->paymentProcessor->process($o))
            ->flatMap(fn($o) => $this->orderRepository->save($o));
    }
}
```

### PHP Anti-Patterns to Avoid

#### ❌ Avoid Mutable State
✅ **Prefer readonly properties**

```php
<?php

declare(strict_types=1);

// BAD - mutable
class Order
{
    public string $orderId;
    public float $totalAmount;
}

// GOOD - immutable with readonly
final readonly class Order
{
    public function __construct(
        public string $orderId,
        public float $totalAmount
    ) {
    }
}
```

#### ❌ Avoid Primitive Obsession
✅ **Always wrap primitives in types**

```php
<?php

declare(strict_types=1);

// BAD - primitive obsession
function createOrder(string $customerId, float $amount): Order
{
    // Easy to pass parameters in wrong order
}

// GOOD - wrapped in types
final readonly class CustomerId
{
    public function __construct(public string $value)
    {
        if (empty($value)) {
            throw new \InvalidArgumentException('CustomerId cannot be empty');
        }
    }
}

function createOrder(CustomerId $customerId, Money $amount): Order
{
    // Type safety - can't mix up parameters
}
```

#### ❌ Avoid Null
✅ **Use Result or explicit nullable types**

```php
<?php

declare(strict_types=1);

// BAD - implicit null
function findCustomer(string $id): Customer
{
    return $this->repository->find($id); // May return null
}

// BETTER - explicit nullable
function findCustomer(string $id): ?Customer
{
    return $this->repository->find($id);
}

// BEST - Result for domain operations
function getCustomer(string $id): Result
{
    $customer = $this->repository->find($id);
    return $customer !== null
        ? Result::success($customer)
        : Result::failure('Customer not found');
}
```

#### ❌ Prefer Result Types Over Exceptions
✅ **Use Result<T> for expected failures**

```php
<?php

declare(strict_types=1);

// BAD - exceptions for control flow
function validateOrder(Order $order): Order
{
    if (empty($order->items)) {
        throw new \InvalidArgumentException('Order has no items');
    }
    return $order;
}

// GOOD - Result type
function validateOrder(Order $order): Result
{
    if (empty($order->items)) {
        return Result::failure('Order has no items');
    }
    return Result::success($order);
}
```

#### ❌ Don't Use declare(strict_types=1) Inconsistently
✅ **Always use strict types**

```php
<?php

// ALWAYS at the top of EVERY file
declare(strict_types=1);

namespace App\Domain;

// Your code here
```

### Naming Conventions

#### File Naming
- **Style**: `PascalCase.php`
- **Rule**: File names **must match** the class/interface name inside
- **Examples**:
  - `OrderService.php` contains `class OrderService`
  - `Customer.php` contains `class Customer`
  - `OrderRepository.php` contains `interface OrderRepository`

#### Test File Naming
- **Format**: `ClassNameTest.php`
- **Examples**:
  - `OrderService.php` → `OrderServiceTest.php`
  - `Customer.php` → `CustomerTest.php`

#### PHP Naming Conventions (PSR Standards)

- **Classes, Interfaces, Traits**: `PascalCase`
  ```php
  class OrderService {}
  interface OrderRepository {}
  trait Timestampable {}
  ```

- **Methods**: `camelCase`
  ```php
  public function calculateTotal(array $items): Money {}
  public function processOrder(Order $order): Result {}
  ```

- **Properties**: `camelCase`
  ```php
  private readonly OrderRepository $orderRepository;
  public string $orderId;
  ```

- **Constants**: `UPPER_SNAKE_CASE`
  ```php
  public const MAX_RETRY_COUNT = 3;
  private const API_BASE_URL = 'https://api.example.com';
  ```

- **Namespaces**: `PascalCase` (PSR-4)
  ```php
  namespace App\Domain\ValueObjects;
  namespace App\Service;
  ```

### Domain Modeling

⚠️ **Ask user confirmation** when modeling domain types
- Discuss type choices (classes, enums, readonly)
- Validate domain vocabulary with user
- Confirm validation and error handling approach

### Dependencies

#### Composer (Package Manager)
✅ **Use Composer for dependency management**

**composer.json**:
```json
{
    "name": "vendor/project",
    "description": "Project description",
    "type": "project",
    "require": {
        "php": "^8.3"
    },
    "require-dev": {
        "phpunit/phpunit": "^10.5",
        "phpstan/phpstan": "^1.10",
        "friendsofphp/php-cs-fixer": "^3.40"
    },
    "autoload": {
        "psr-4": {
            "App\\": "src/"
        }
    },
    "autoload-dev": {
        "psr-4": {
            "Tests\\": "tests/"
        }
    }
}
```

#### Testing Framework
✅ **PHPUnit** - Standard testing framework

```php
<?php

declare(strict_types=1);

namespace Tests\Unit;

use PHPUnit\Framework\TestCase;
use App\Service\OrderService;
use App\Domain\Order;

final class OrderServiceTest extends TestCase
{
    public function testShouldValidateOrderSuccessfully(): void
    {
        // Given
        $order = new Order(
            id: new OrderId('ORD-001'),
            customer: new Customer('John Doe'),
            items: [new OrderItem('item1', Money::usd(10.0))],
            status: OrderStatus::PENDING
        );
        $service = new OrderService();

        // When
        $result = $service->validateOrder($order);

        // Then
        $this->assertTrue($result->isSuccess());
        $this->assertEquals($order, $result->value);
    }
}
```

#### Code Quality Tools
✅ **Use static analysis and code style tools**

```bash
# PHPStan - Static analysis
./vendor/bin/phpstan analyse src tests

# PHP CS Fixer - Code style
./vendor/bin/php-cs-fixer fix

# PHPUnit - Tests with coverage
./vendor/bin/phpunit --coverage-html coverage/
```

---

## Git Workflow

### Git Command Rules

1. ❌ **NEVER** use `git stash` / `git push` or its alias `psh`
2. ❌ **NEVER** use `--force` or `--force-with-lease` options
3. ⚠️ Before running a new git command not listed here, **ask to add it to the allow list**
4. ✅ Use git freely for checking differences (`git diff`, `git log`, `git status`)
5. ✅ Commit only when tests pass with assertions AND code coverage is verified

### Allowed Git Commands

```bash
git status
git diff
git log
git add <files>
git commit -m "message"
# Add more commands here as approved
```

### Commit Message Format

All commit messages must start with one of these prefixes:

- `feat(agt):` - New feature or functionality
- `fix(agt):` - Bug fixes in the codebase
- `refac(agt):` - Refactoring (preparing or finishing a feature)
- `chore(agt):` - Cleanup, removing dead code
- `doc(agt):` - Documentation changes (.md files)
- `test(agt):` - When touching test only

**Examples:**
```
feat(agt): add order validation service
fix(agt): correct tax calculation in payment processor
refac(agt): extract common validation logic to base class
chore(agt): remove deprecated API endpoints
doc(agt): update README with new API examples
```

### Commit Process

1. Run all tests (including integration tests)
2. Verify 90% code coverage minimum
3. **Ask permission to commit**
4. **Wait for user review**
5. Only commit after approval

---

## When Stuck or Uncertain

### Decision Matrix

| Situation | Action |
|-----------|--------|
| **Architecture questions** | Ask user with pros/cons example |
| **Domain modeling** | Ask user for confirmation with proposed types |
| **File location unclear** | Ask user where to create new files |
| **New git command needed** | Ask to add to allow list |
| **Design complexity** | Use top-down approach with mocks (`throw new \LogicException()`) |
| **Test strategy unclear** | Start with acceptance tests (domain level) |
| **Coverage below 90%** | Add more test cases before committing |
| **Tests failing** | Do NOT commit - fix tests first |

### General Principle

**When in doubt, ask the user.** Collaboration is essential.

---

## Summary Checklist

Before any code modification:
- [ ] Have you read the README.md and CONTRIBUTING.md?
- [ ] Have you created and discussed a todo-list with the user?
- [ ] Have you received approval to proceed?

During development:
- [ ] Are you writing tests first (TDD)?
- [ ] Are you using `declare(strict_types=1)` in every file?
- [ ] Are you using Result pattern for error handling?
- [ ] Are you using readonly for immutable data?
- [ ] Are you wrapping primitives in types?
- [ ] Are you using enums for closed sets?
- [ ] Are you using named arguments for clarity?
- [ ] Are you following PSR standards?

Before committing:
- [ ] Do all tests pass with meaningful assertions?
- [ ] Is code coverage at least 90%?
- [ ] Have you asked permission to commit?
- [ ] Have you received user approval?
- [ ] Is your commit message properly formatted with (agt) prefix?

---

**Last Updated:** 2025-12-19
**Status:** Complete - PHP AI Agent Guidelines
