# Perl Project Template

This directory contains a complete, autonomous AI Agent configuration for Perl projects.

## For AI Agents

**Start here**: Read the [AGENTS_RULES.md](./AGENTS_RULES.md) file for complete rules.

All necessary guidelines are in this directory:
- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete rules (works with all AI assistants)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE (auto-loaded)
- **[AGENTS.md](./AGENTS.md)** - Quick reference
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guide

## For Developers

When starting a new Perl project in this directory:

1. Follow the Quick Start guide below
2. Ensure the AI agent reads the `.cursorrules` file
3. Follow TDD and Domain-Driven Design principles

## Structure

- `.cursorrules` - Complete AI agent rules (auto-loaded by Cursor)
- `AGENTS.md` - Quick reference and links to full documentation
- `README.md` - This file

## Key Requirements

- **Test-Driven Development**: Write tests first
- **90% Code Coverage**: Minimum requirement
- **Domain-Driven Design**: Business vocabulary first
- **Modern Perl**: Use Perl 5.38+ with modern features
- **Moose/Moo**: Object-oriented programming
- **Type::Tiny**: Type constraints and validation
- **Result Pattern**: Use Result types for error handling
- **Immutability**: Use readonly attributes by default

## Getting Started

```bash
# Create project directory
mkdir ProjectName
cd ProjectName

# Create standard directory structure
mkdir -p lib/ProjectName
mkdir -p t/unit
mkdir -p t/integration
mkdir -p t/acceptance
mkdir -p bin
mkdir -p docs

# Create cpanfile for dependencies
cat > cpanfile << 'EOF'
requires 'perl', '5.038';

# Core dependencies
requires 'Moose', '>= 2.2200';
requires 'namespace::autoclean', '>= 0.29';
requires 'Try::Tiny', '>= 0.31';
requires 'Type::Tiny', '>= 2.004000';

# Testing
on 'test' => sub {
    requires 'Test::More', '>= 1.302195';
    requires 'Test::Exception', '>= 0.43';
    requires 'Test::Deep', '>= 1.130';
    requires 'Test::MockObject', '>= 1.20200122';
    requires 'Devel::Cover', '>= 1.40';
};
EOF

# Install dependencies
cpanm --installdeps .

# Or with carton
cpanm Carton
carton install

# Initialize git
git init
```

## Example Module Structure

```perl
# lib/ProjectName/Order.pm
package ProjectName::Order;

use v5.38;                    # Modern Perl features
use warnings FATAL => 'all';
use namespace::autoclean;
use feature 'signatures';
no warnings 'experimental::signatures';

use Moose;
use MooseX::StrictConstructor;

# Attributes
has 'order_id' => (
    is       => 'ro',
    isa      => 'Str',
    required => 1,
);

has 'items' => (
    is      => 'ro',
    isa     => 'ArrayRef',
    default => sub { [] },
);

# Methods
sub add_item ($self, $item) {
    push @{$self->items}, $item;
    return $self;
}

__PACKAGE__->meta->make_immutable;
1;
```

## Example Test Structure

```perl
# t/unit/order.t
use strict;
use warnings;
use Test::More;
use Test::Exception;

use ProjectName::Order;

subtest 'create order with valid data' => sub {
    my $order = ProjectName::Order->new(
        order_id => 'ORD-123456',
        items    => [],
    );
    
    ok defined $order, 'order created';
    is $order->order_id, 'ORD-123456', 'order_id set correctly';
};

done_testing;
```

## Running Tests

```bash
# Run all tests
prove -l t/

# Run with verbose output
prove -lv t/

# Run specific test
prove -lv t/unit/order.t

# Run with coverage
cover -delete
PERL5OPT=-MDevel::Cover prove -l t/
cover

# Generate HTML coverage report
cover -report html

# Check coverage threshold
cover -report text -coverage_threshold 90
```

---

## Complete Documentation

All documentation is self-contained in this directory:

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (universal)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines
- **[README.md](./README.md)** - This file

**Note**: This directory is completely autonomous and works with any AI assistant (Cursor, GitHub Copilot, ChatGPT, Claude, etc.).
