# Code Quality Principles

**⚠️ CRITICAL: Read these principles before code reviews and refactoring**

## 1. Avoid Bad Trade-offs with Default Values

### ❌ **Problem**: Creating default/fallback values to mask missing data

```perl
# BAD: Hiding potential issues with defaults
my $minimal_rate_plan = $room_stay_price->{rate_plans}[0] // {
    rate_plan_code => undef,
    commission => { percent => 0, status_type => 'full' },
    cancel_penalty => { cancel_deadline => undef, is_non_refundable => 0 },
    # ... many more defaults
};
```

### ✅ **Solution**: Fail fast with proper error handling

```perl
# GOOD: Explicit error handling, no hidden defaults
my $rate_plan = $room_stay_price->{rate_plans}[0];

unless (defined $rate_plan) {
    return ProjectName::Result->err("No rate plan found");
}

return ProjectName::Result->ok($rate_plan);
```

```perl
# GOOD: Explicit failure to mock and decide later
my $guarantee_for_check = die "TODO(agt): Properly convert GuaranteeTerm to Guarantee";

# BAD: Hides with a default value trade-offs
# TODO(agt): Properly convert GuaranteeTerm to Guarantee
my $guarantee_for_check = undef;
```

### **Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ❌ Hides problems until production
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling
- ✅ Consistent error propagation

## 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution. When you encounter similar scenarios in the codebase, use the same approach to handle them.

### Why it matters:
- ✅ Makes code predictable and easier to understand
- ✅ Reduces cognitive load when reading code
- ✅ Makes refactoring safer
- ✅ Helps identify patterns and abstractions

## 3. Extract Reusable Functions to Modules

Organize code for reuse. When you see repeated logic, extract it into a well-named subroutine in an appropriate module.

### Why it matters:
- ✅ Reduces code duplication
- ✅ Makes changes easier (single point of modification)
- ✅ Improves testability
- ✅ Creates a vocabulary of reusable operations
