# AI Agent Rules for Perl Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on Perl projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
# Run all tests
prove -l t/

# Run with coverage
cover -delete
PERL5OPT=-MDevel::Cover prove -l t/
cover

# Coverage threshold check (90% minimum)
cover -report text -coverage_threshold 90
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check Devel::Cover reports for new/modified code (minimum 90% coverage)

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Domain-Driven Design & Test Types

**Acceptance Tests (Domain-Focused)**
- **Focus**: Business rules, domain input/output
- **Purpose**: Test business behavior, not implementation details
- **Robustness**: Should survive refactoring (high-level, stable contracts)
- **Vocabulary**: Use business/domain terminology
- **When**: Test features from user/business perspective
- **Location**: `t/acceptance/` or `t/domain/`

**Unit Tests (Technical-Focused)**
- **Focus**: Simple, isolated components (business OR technical)
- **Purpose**: Test individual methods, functions, patterns
- **Scope**: Lower-level than acceptance tests
- **Location**: `t/unit/`

**Integration Tests**
- **Focus**: Multiple components working together
- **Purpose**: Test external service integration, database access
- **Location**: `t/integration/`

#### Separation of Concerns
- **Business rules** vs **Technical rules** must be clearly separated
- Domain-Driven Design: Start with business problems and vocabulary first
- Technical code (algorithms, patterns) implements the domain model

#### The "Red" Phase Matters
1. **Design First**: The "red" phase validates design at compile time
2. **Top-Down Approach**: When appropriate, implement design by:
   - Defining method signatures with correct input/output types
   - Using business vocabulary in class/method names
   - Stubbing methods with `die "Not implemented yet"`
   - **Code must compile** even with stubs
   - Tests stay red but design is validated
3. **Design Quality Indicator**: If underlying parts are easy to implement, the design is good

#### Make It Work, Then Make It Right
- When implementing features that require refactoring:
  1. Make it work (get tests passing)
  2. Make it right (refactor without changing acceptance tests)
- **Acceptance tests should NOT change during refactoring**
- This proves the refactoring preserves business behavior

### Test Commands

```bash
# Run ALL tests (TDD approach - includes integration tests)
prove -l t/
prove -lv t/           # Verbose output

# Run specific test file
prove -lv t/unit/order_service.t

# Run with coverage (Devel::Cover)
cover -delete
PERL5OPT=-MDevel::Cover prove -l t/
cover

# Generate HTML coverage report
cover -report html

# Coverage threshold check (90% minimum)
cover -report text -coverage_threshold 90

# Run with Test::Harness
make test              # If using Makefile.PL
./Build test          # If using Build.PL

# Run specific test categories
prove -lv t/unit/
prove -lv t/integration/
prove -lv --state=failed  # Re-run only failed tests
```

**Note for WSL users**: Replace `perl` with `perl.exe` and `prove` with `prove.exe` if targeting Windows Perl.

---

## Development

### Build Environment

**Important for WSL users**: When running on WSL (Windows Subsystem for Linux), always use `perl.exe` instead of `perl` to ensure you're using the Windows Perl installation if that's your target environment.

### Build Commands

```bash
# Using Makefile.PL
perl Makefile.PL
make
make test
make install

# Using Build.PL (Module::Build)
perl Build.PL
./Build
./Build test
./Build install

# With carton (recommended for development)
carton install          # Install dependencies
carton exec perl script.pl  # Run with bundled deps
```

### Code Quality Principles

#### 1. Avoid Bad Trade-offs with Default Values

**❌ Problem**: Creating default/fallback values to mask missing data

**✅ Solution**: Fail fast with proper error handling

```perl
# GOOD: Explicit failure to mock and decide later
my $guarantee_for_check = die "TODO(agt): Properly convert GuaranteeTerm to Guarantee";

# BAD: Hides with a default value trade-offs
# TODO(agt): Properly convert GuaranteeTerm to Guarantee
my $guarantee_for_check = undef;
```

**Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling

#### 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution

#### 3. Extract Reusable Functions to Modules

Organize for reuse

### TODO Comments

When adding todo in the code base, use this prefix: `#TODO(agt)`

---

## Perl Specific Guidelines

### Modern Perl Preamble

✅ **Always start modules with modern Perl features**

```perl
package ProjectName::Order;

use v5.38;                    # Enables strict, warnings, say, signatures
use warnings FATAL => 'all';  # Make all warnings fatal
use namespace::autoclean;     # Clean up imported symbols
use feature 'signatures';     # Enable subroutine signatures
no warnings 'experimental::signatures';

# Class definition with Moose
use Moose;
use MooseX::StrictConstructor;  # Reject unknown constructor arguments

# Your code here

__PACKAGE__->meta->make_immutable;  # Optimize Moose class
1;
```

### Required Patterns

#### Object System: Moose (or Moo)

✅ **Use Moose for object-oriented programming**

**Moose (full-featured, slower but powerful):**
```perl
package ProjectName::Order;
use Moose;
use namespace::autoclean;

# Attributes with type constraints
has 'order_id' => (
    is       => 'ro',           # read-only
    isa      => 'Str',          # type constraint
    required => 1,              # mandatory
);

has 'customer' => (
    is       => 'ro',
    isa      => 'ProjectName::Customer',
    required => 1,
);

has 'items' => (
    is      => 'ro',
    isa     => 'ArrayRef[ProjectName::OrderItem]',
    default => sub { [] },
);

has 'total' => (
    is      => 'ro',
    isa     => 'Num',
    lazy    => 1,              # Compute on first access
    builder => '_build_total',
);

# Methods
sub _build_total {
    my $self = shift;
    my $sum = 0;
    $sum += $_->price * $_->quantity for @{$self->items};
    return $sum;
}

sub add_item {
    my ($self, $item) = @_;
    push @{$self->items}, $item;
    return $self;
}

__PACKAGE__->meta->make_immutable;
1;
```

**Moo (lightweight alternative, faster):**
```perl
package ProjectName::Order;
use Moo;
use Types::Standard qw( Str ArrayRef InstanceOf Num );
use namespace::autoclean;

has 'order_id' => (
    is       => 'ro',
    isa      => Str,
    required => 1,
);

has 'items' => (
    is      => 'ro',
    isa     => ArrayRef[InstanceOf['ProjectName::OrderItem']],
    default => sub { [] },
);

has 'total' => (
    is      => 'ro',
    isa     => Num,
    lazy    => 1,
    builder => '_build_total',
);

# Methods same as Moose

1;
```

#### Type Constraints with Type::Tiny

✅ **Use Type::Tiny for robust type checking**

```perl
package ProjectName::Types;
use Type::Library -base;
use Type::Utils -all;
use Types::Standard qw( Str Num ArrayRef Dict Optional );

# Custom types
declare "OrderId",
    as Str,
    where { /^ORD-\d{6}$/ },
    message { "OrderId must be in format ORD-123456" };

declare "PositiveNum",
    as Num,
    where { $_ > 0 },
    message { "Number must be positive" };

declare "Email",
    as Str,
    where { /^[^@]+@[^@]+\.[^@]+$/ },
    message { "Invalid email format" };

# Coercions
declare "OrderTotal",
    as PositiveNum;

coerce "OrderTotal",
    from Str,
    via { 0+ $_ };  # String to number

1;

# Usage in class
package ProjectName::Order;
use Moose;
use ProjectName::Types qw( OrderId PositiveNum );

has 'order_id' => (
    is  => 'ro',
    isa => OrderId,
);

has 'total' => (
    is     => 'ro',
    isa    => PositiveNum,
    coerce => 1,
);
```

#### Result Pattern for Error Handling

✅ **Use Result types instead of exceptions**

```perl
package ProjectName::Result;
use Moose;
use namespace::autoclean;

has 'success' => (is => 'ro', isa => 'Bool', required => 1);
has 'value'   => (is => 'ro');
has 'error'   => (is => 'ro', isa => 'Str');

sub ok {
    my ($class, $value) = @_;
    return $class->new(success => 1, value => $value);
}

sub err {
    my ($class, $error) = @_;
    return $class->new(success => 0, error => $error);
}

sub is_ok  { $_[0]->success }
sub is_err { !$_[0]->success }

sub unwrap {
    my $self = shift;
    die $self->error unless $self->success;
    return $self->value;
}

sub unwrap_or {
    my ($self, $default) = @_;
    return $self->success ? $self->value : $default;
}

__PACKAGE__->meta->make_immutable;
1;

# Usage
package ProjectName::Service::OrderService;

sub validate_order {
    my ($self, $order) = @_;
    
    return ProjectName::Result->err("Order has no items")
        if @{$order->items} == 0;
    
    return ProjectName::Result->err("Total is negative")
        if $order->total < 0;
    
    return ProjectName::Result->ok($order);
}

sub process_order {
    my ($self, $order) = @_;
    
    my $result = $self->validate_order($order);
    return $result if $result->is_err;
    
    # Continue processing...
    return ProjectName::Result->ok($processed_order);
}
```

#### Try::Tiny for Exception Handling

✅ **Use Try::Tiny for exception handling when needed**

```perl
use Try::Tiny;

sub process_payment {
    my ($self, $order) = @_;
    
    try {
        my $response = $self->payment_gateway->charge($order->total);
        return ProjectName::Result->ok($response);
    }
    catch {
        my $error = $_;
        $self->logger->error("Payment failed: $error");
        return ProjectName::Result->err("Payment processing failed");
    };
}
```

#### Method Signatures (Perl 5.36+)

✅ **Use subroutine signatures for cleaner code**

```perl
use v5.38;  # Enables signatures by default
use feature 'signatures';
no warnings 'experimental::signatures';

sub calculate_discount ($self, $customer, $amount) {
    return $amount * 0.20 if $customer->is_premium && $customer->years_active > 5;
    return $amount * 0.15 if $customer->is_premium;
    return $amount * 0.10 if $customer->years_active > 10;
    return $amount * 0.05 if $customer->years_active > 5;
    return 0;
}

# With defaults
sub create_order ($self, $customer, $items = []) {
    # ...
}

# Named parameters (using hashref)
sub create_order ($self, $args) {
    my $customer = $args->{customer} // die "customer required";
    my $items    = $args->{items} // [];
    # ...
}
```

#### Roles for Code Reuse

✅ **Use Moose Roles for shared behavior**

```perl
package ProjectName::Role::Validatable;
use Moose::Role;
use namespace::autoclean;

requires 'validate';  # Classes must implement this

sub is_valid {
    my $self = shift;
    my $result = $self->validate;
    return $result->is_ok;
}

1;

# Usage
package ProjectName::Order;
use Moose;
with 'ProjectName::Role::Validatable';

sub validate {
    my $self = shift;
    
    return ProjectName::Result->err("No items")
        if @{$self->items} == 0;
    
    return ProjectName::Result->ok($self);
}
```

### Perl Anti-Patterns to Avoid

#### ❌ Avoid Direct Package Variables
✅ **Use Moose attributes or lexical variables**

```perl
# BAD - package variables
package MyClass;
our $counter = 0;

sub increment {
    $counter++;
}

# GOOD - lexical variable (for non-OO)
{
    my $counter = 0;
    
    sub increment {
        $counter++;
    }
    
    sub get_counter {
        return $counter;
    }
}

# BETTER - Moose attribute
package MyClass;
use Moose;

has 'counter' => (
    is      => 'rw',
    isa     => 'Int',
    default => 0,
);

sub increment {
    my $self = shift;
    $self->counter($self->counter + 1);
}
```

#### ❌ Avoid Primitive Obsession
✅ **Always wrap primitives in types**

```perl
# BAD - primitive obsession
sub create_order {
    my ($customer_id, $amount, $currency) = @_;
    # Easy to mix up parameters
    # No validation
}

# GOOD - wrapped in value objects
package ProjectName::ValueObject::CustomerId;
use Moose;
has 'value' => (is => 'ro', isa => 'Str', required => 1);

package ProjectName::ValueObject::Money;
use Moose;
has 'amount'   => (is => 'ro', isa => 'Num', required => 1);
has 'currency' => (is => 'ro', isa => 'Str', default => 'USD');

sub create_order {
    my ($customer_id, $money) = @_;
    # Type safety
    # Built-in validation
}
```

#### ❌ Avoid Bareword Filehandles
✅ **Use lexical filehandles**

```perl
# BAD - bareword filehandle
open(FILE, '<', 'data.txt') or die $!;
my $content = <FILE>;
close(FILE);

# GOOD - lexical filehandle
open my $fh, '<', 'data.txt' or die $!;
my $content = <$fh>;
close $fh;

# BETTER - Path::Tiny
use Path::Tiny;
my $content = path('data.txt')->slurp;
```

#### ❌ Avoid Indirect Object Syntax
✅ **Use direct method calls**

```perl
# BAD - indirect object syntax
my $obj = new MyClass;          # Can be ambiguous
my $result = compute $obj 42;   # Hard to parse

# GOOD - direct method calls
my $obj = MyClass->new;
my $result = $obj->compute(42);
```

#### ❌ Don't Use Exceptions for Control Flow
✅ **Use Result types for expected failures**

```perl
# BAD - exceptions for control flow
sub validate_order {
    my $order = shift;
    die "No items" if @{$order->items} == 0;
    return $order;
}

# GOOD - Result type
sub validate_order {
    my $order = shift;
    
    return ProjectName::Result->err("No items")
        if @{$order->items} == 0;
    
    return ProjectName::Result->ok($order);
}

# Exceptions are still appropriate for:
# - Programming errors (die "Required parameter missing")
# - Unrecoverable errors
# - Infrastructure failures
```

#### ❌ Avoid Global State
✅ **Inject dependencies**

```perl
# BAD - global state
our $DATABASE;

sub get_order {
    my $id = shift;
    return $DATABASE->select_order($id);
}

# GOOD - dependency injection
package MyService;
use Moose;

has 'database' => (
    is       => 'ro',
    does     => 'ProjectName::Role::Database',
    required => 1,
);

sub get_order {
    my ($self, $id) = @_;
    return $self->database->select_order($id);
}
```

#### ❌ Avoid Hash-Based Objects
✅ **Use Moose/Moo instead of blessed hashes**

```perl
# BAD - manual blessed hash
package Order;

sub new {
    my $class = shift;
    my %args = @_;
    return bless \%args, $class;
}

sub order_id {
    my $self = shift;
    return $self->{order_id};
}

# GOOD - Moose
package Order;
use Moose;

has 'order_id' => (is => 'ro', isa => 'Str', required => 1);

__PACKAGE__->meta->make_immutable;
```

### Naming Conventions

#### File Naming
- **Style**: `CamelCase.pm` for modules
- **Rule**: File path **must match** the package namespace
- **Examples**:
  - `lib/ProjectName/Order.pm` contains `package ProjectName::Order`
  - `lib/ProjectName/Service/OrderService.pm` contains `package ProjectName::Service::OrderService`

#### Test File Naming
- **Format**: `feature_name.t` (lowercase with underscores)
- **Examples**:
  - `Order.pm` → `t/unit/order.t`
  - `OrderService.pm` → `t/unit/service/order_service.t`

#### Perl Naming Conventions (Community Standards)

- **Packages/Classes**: `CamelCase`
  ```perl
  package ProjectName::Order;
  package ProjectName::Service::OrderService;
  ```

- **Subroutines/Methods**: `snake_case`
  ```perl
  sub calculate_total { }
  sub process_order { }
  sub get_customer_by_id { }
  ```

- **Variables**: `snake_case` (lowercase with underscores)
  ```perl
  my $order_id = '12345';
  my $total_amount = 100.00;
  my @order_items = ();
  my %customer_data = ();
  ```

- **Constants**: `UPPER_CASE`
  ```perl
  use constant MAX_RETRY_COUNT => 3;
  use constant API_BASE_URL    => 'https://api.example.com';
  
  # Or with Const::Fast (preferred)
  use Const::Fast;
  const my $MAX_RETRY_COUNT => 3;
  ```

- **Private Methods**: Prefix with `_`
  ```perl
  sub _internal_helper { }
  sub _validate_input { }
  ```

- **Accessors/Mutators**: `attribute_name` (via Moose/Moo)
  ```perl
  has 'order_id'   => (is => 'ro', isa => 'Str');
  has 'total'      => (is => 'rw', isa => 'Num');
  ```

### Domain Modeling

⚠️ **Ask user confirmation** when modeling domain types
- Discuss class/type choices (Moose classes, value objects, roles)
- Validate domain vocabulary with user
- Confirm validation and error handling approach

### Dependencies

#### Key Libraries

**Object System**
✅ **Moose** - Full-featured OO system (recommended)
✅ **Moo** - Lightweight alternative

**Type System**
✅ **Type::Tiny** - Type constraints and coercions

**Error Handling**
✅ **Try::Tiny** - Exception handling

**Utilities**
✅ **List::Util** - List processing (core module)
✅ **Path::Tiny** - File operations
✅ **JSON::MaybeXS** - JSON handling (fastest available)
✅ **Const::Fast** - Constants

**Testing**
✅ **Test::More** - Core testing (bundled with Perl)
✅ **Test2::V0** - Modern test suite
✅ **Test::Exception** - Exception testing
✅ **Test::Deep** - Deep data structure testing
✅ **Test::MockObject** - Mocking
✅ **Devel::Cover** - Code coverage

---

## Git Workflow

### Git Command Rules

1. ❌ **NEVER** use `git stash` / `git push` or its alias `psh`
2. ❌ **NEVER** use `--force` or `--force-with-lease` options
3. ⚠️ Before running a new git command not listed here, **ask to add it to the allow list**
4. ✅ Use git freely for checking differences (`git diff`, `git log`, `git status`)
5. ✅ Commit only when tests pass with assertions AND code coverage is verified

### Allowed Git Commands

```bash
git status
git diff
git log
git add <files>
git commit -m "message"
# Add more commands here as approved
```

### Commit Message Format

All commit messages must start with one of these prefixes:

- `feat(agt):` - New feature or functionality
- `fix(agt):` - Bug fixes in the codebase
- `refac(agt):` - Refactoring (preparing or finishing a feature)
- `chore(agt):` - Cleanup, removing dead code
- `doc(agt):` - Documentation changes (.md files)
- `test(agt):` - When touching test only

**Examples:**
```
feat(agt): add Order validation with Type::Tiny
fix(agt): correct tax calculation in payment processor
refac(agt): extract common validation to Role
chore(agt): remove deprecated API endpoints
doc(agt): update README with new testing examples
```

### Commit Process

1. Run all tests (including integration tests)
2. Verify 90% code coverage minimum
3. **Ask permission to commit**
4. **Wait for user review**
5. Only commit after approval

---

## Project Structure & Organization

### Standard Directory Layout

```
ProjectName/
├── lib/
│   └── ProjectName/
│       ├── Domain/
│       │   ├── Order.pm
│       │   ├── Customer.pm
│       │   └── ValueObject/
│       │       ├── OrderId.pm
│       │       └── Money.pm
│       ├── Service/
│       │   ├── OrderService.pm
│       │   └── PaymentService.pm
│       ├── Repository/
│       │   └── OrderRepository.pm
│       ├── Role/
│       │   └── Validatable.pm
│       └── Types.pm
├── t/
│   ├── unit/
│   │   ├── domain/
│   │   │   └── order.t
│   │   └── service/
│   │       └── order_service.t
│   ├── integration/
│   │   └── database.t
│   ├── acceptance/
│   │   └── order_workflow.t
│   └── data/
│       └── sample_data.json
├── bin/
│   └── process_orders.pl
├── docs/
│   └── api.md
├── cpanfile
├── Makefile.PL
├── MANIFEST.SKIP
├── Changes
└── README.md
```

### Project Boundaries

#### Never Modify
❌ **Do not modify or commit:**
- `blib/` - Build library directory
- `cover_db/` - Coverage database
- `MANIFEST.bak` - Backup manifest
- `Makefile` - Generated makefile
- `MYMETA.*` - Generated meta files
- `local/` - Carton local dependencies
- `*.o`, `*.bs` - Binary files
- Any other build artifacts

✅ **Ensure `.gitignore` is configured** to exclude these files

#### Can Modify (After Planning)
✅ **Can be modified** (after todo-list approval):
- `*.pm` files - Module files
- `*.pl` files - Script files
- `*.t` files - Test files
- `cpanfile` - Dependency file
- `Makefile.PL` / `Build.PL` - Build configuration
- `MANIFEST` - File list

**Remember**: The planning phase ensures all modifications are discussed before implementation.

---

## Quick Start for New Projects

### 1. Create Project Structure

```bash
# Create project directory
mkdir ProjectName
cd ProjectName

# Create standard directory structure
mkdir -p lib/ProjectName
mkdir -p t
mkdir -p bin
mkdir -p docs

# Initialize git
git init
```

### 2. Create Build System Files

#### Create `cpanfile` for dependencies

```perl
# cpanfile
requires 'perl', '5.038';

# Core dependencies
requires 'Moose', '>= 2.2200';
requires 'namespace::autoclean', '>= 0.29';
requires 'Try::Tiny', '>= 0.31';
requires 'Type::Tiny', '>= 2.004000';

# Testing
on 'test' => sub {
    requires 'Test::More', '>= 1.302195';
    requires 'Test::Exception', '>= 0.43';
    requires 'Test::Deep', '>= 1.130';
    requires 'Test::MockObject', '>= 1.20200122';
    requires 'Devel::Cover', '>= 1.40';
};
```

### 3. Install Dependencies

```bash
# Install cpanminus if not available
curl -L https://cpanmin.us | perl - App::cpanminus

# Install dependencies from cpanfile
cpanm --installdeps .

# Or with carton (bundler for Perl)
cpanm Carton
carton install
```

### 4. Create .gitignore

```
# Perl
blib/
.build/
_build/
cover_db/
inc/
Build
Build.bat
.last_cover_stats
Makefile
Makefile.old
MANIFEST.bak
META.yml
META.json
MYMETA.*
nytprof.out
pm_to_blib
*.o
*.bs
*.swp
*.bak

# Carton
local/
.carton/

# IDE
.vscode/
.idea/
*.iml
```

---

## When Stuck or Uncertain

### Decision Matrix

| Situation | Action |
|-----------|--------|
| **Architecture questions** | Ask user with pros/cons example |
| **Domain modeling** | Ask user for confirmation with proposed types |
| **File location unclear** | Ask user where to create new files |
| **New git command needed** | Ask to add to allow list |
| **Design complexity** | Use top-down approach with stubs (`die "Not implemented"`) |
| **Test strategy unclear** | Start with acceptance tests (domain level) |
| **Coverage below 90%** | Add more test cases before committing |
| **Tests failing** | Do NOT commit - fix tests first |
| **Moose vs Moo** | Ask user preference (Moose = full features, Moo = lightweight) |

### General Principle

**When in doubt, ask the user.** Collaboration is essential.

---

## Summary Checklist

Before any code modification:
- [ ] Have you read the README.md and CONTRIBUTING.md?
- [ ] Have you created and discussed a todo-list with the user?
- [ ] Have you received approval to proceed?

During development:
- [ ] Are you writing tests first (TDD)?
- [ ] Are you using Result pattern for error handling?
- [ ] Are you using Moose/Moo for OO?
- [ ] Are you wrapping primitives in types?
- [ ] Are you using Type::Tiny for type constraints?
- [ ] Are you avoiding global state and package variables?

Before committing:
- [ ] Do all tests pass with meaningful assertions?
- [ ] Is code coverage at least 90%?
- [ ] Have you asked permission to commit?
- [ ] Have you received user approval?
- [ ] Is your commit message properly formatted with (agt) prefix?

---

**Last Updated:** 2025-12-19
**Status:** Complete - Perl AI Agent Guidelines
