# Kotlin Project Template

This directory contains a complete, autonomous AI Agent configuration for Kotlin projects.

## For AI Agents

**Start here**: Read the [AGENTS_RULES.md](./AGENTS_RULES.md) file for complete rules.

All necessary guidelines are in this directory:
- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete rules (works with all AI assistants)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE (auto-loaded)
- **[AGENTS.md](./AGENTS.md)** - Quick reference
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guide

## For Developers

When starting a new Kotlin project in this directory:

1. Follow the Quick Start guide below
2. Ensure the AI agent reads the `AGENTS_RULES.md` file
3. Follow TDD and Domain-Driven Design principles

## Structure

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (universal)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines
- **[README.md](./README.md)** - This file

## Key Requirements

- **Test-Driven Development**: Write tests first
- **90% Code Coverage**: Minimum requirement
- **Domain-Driven Design**: Business vocabulary first
- **Result Pattern**: Use Result types for error handling
- **Immutability**: Use val and data classes
- **Type Safety**: Use value classes, leverage null safety
- **Modern Kotlin**: Use Kotlin 1.9+ features
- **Coroutines**: For asynchronous operations

## Getting Started

### With Gradle (Recommended)

```bash
# Create a new Kotlin project with Gradle
gradle init --type kotlin-application

cd my-project

# Update build.gradle.kts for Kotlin
# Add JUnit 5, Kotest, MockK, JaCoCo

# Build and test
./gradlew clean build
./gradlew test jacocoTestReport
```

### build.gradle.kts Example

```kotlin
plugins {
    kotlin("jvm") version "2.0.0"
    id("jacoco")
}

group = "com.example"
version = "1.0.0"

repositories {
    mavenCentral()
}

dependencies {
    implementation(kotlin("stdlib"))
    implementation("org.jetbrains.kotlinx:kotlinx-coroutines-core:1.8.0")
    
    testImplementation(kotlin("test"))
    testImplementation("io.kotest:kotest-runner-junit5:5.8.0")
    testImplementation("io.kotest:kotest-assertions-core:5.8.0")
    testImplementation("io.mockk:mockk:1.13.8")
}

tasks.test {
    useJUnitPlatform()
    finalizedBy(tasks.jacocoTestReport)
}

tasks.jacocoTestReport {
    dependsOn(tasks.test)
    reports {
        xml.required.set(true)
        html.required.set(true)
    }
}

kotlin {
    jvmToolchain(21)
}
```

## Example Project Structure

```
my-project/
├── src/
│   ├── main/
│   │   └── kotlin/
│   │       └── com/
│   │           └── example/
│   │               ├── domain/
│   │               │   ├── Order.kt
│   │               │   ├── Customer.kt
│   │               │   └── valueobjects/
│   │               │       ├── OrderId.kt
│   │               │       └── Money.kt
│   │               ├── service/
│   │               │   ├── OrderService.kt
│   │               │   └── PaymentService.kt
│   │               └── repository/
│   │                   └── OrderRepository.kt
│   └── test/
│       └── kotlin/
│           └── com/
│               └── example/
│                   ├── domain/
│                   │   └── OrderTest.kt
│                   └── service/
│                       └── OrderServiceTest.kt
├── build.gradle.kts
└── README.md
```

## Example Code

### Domain Model (Data Class)

```kotlin
package com.example.domain

import java.math.BigDecimal

data class Order(
    val id: OrderId,
    val customer: Customer,
    val items: List<OrderItem>,
    val status: OrderStatus
) {
    init {
        require(items.isNotEmpty()) { "Order must have at least one item" }
    }
    
    fun calculateTotal(): BigDecimal = items.sumOf { it.total }
}
```

### Value Object (Value Class)

```kotlin
package com.example.domain.valueobjects

@JvmInline
value class OrderId(val value: String) {
    init {
        require(value.isNotBlank()) { "OrderId cannot be blank" }
    }
}

data class Money(val amount: BigDecimal, val currency: Currency) {
    companion object {
        fun usd(amount: BigDecimal) = Money(amount, Currency.getInstance("USD"))
    }
}
```

### Result Pattern

```kotlin
package com.example.common

sealed class Result<out T> {
    data class Success<T>(val value: T) : Result<T>()
    data class Failure(val error: String) : Result<Nothing>()
    
    fun <U> map(mapper: (T) -> U): Result<U> = when (this) {
        is Success -> Success(mapper(value))
        is Failure -> this
    }
    
    fun <U> flatMap(mapper: (T) -> Result<U>): Result<U> = when (this) {
        is Success -> mapper(value)
        is Failure -> this
    }
    
    companion object {
        fun <T> success(value: T): Result<T> = Success(value)
        fun <T> failure(error: String): Result<T> = Failure(error)
    }
}
```

### Service with Coroutines

```kotlin
package com.example.service

import com.example.common.Result
import com.example.domain.Order
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

class OrderService(
    private val orderRepository: OrderRepository,
    private val paymentProcessor: PaymentProcessor
) {
    suspend fun validateOrder(order: Order): Result<Order> = withContext(Dispatchers.Default) {
        when {
            order.items.isEmpty() -> Result.failure("Order has no items")
            order.calculateTotal() <= BigDecimal.ZERO -> Result.failure("Order total must be positive")
            else -> Result.success(order)
        }
    }
    
    suspend fun processOrder(order: Order): Result<Order> {
        return validateOrder(order)
            .flatMap { paymentProcessor.process(it) }
            .flatMap { orderRepository.save(it) }
    }
}
```

### Test Example (Kotest)

```kotlin
package com.example.service

import io.kotest.core.spec.style.DescribeSpec
import io.kotest.matchers.shouldBe
import io.kotest.matchers.types.shouldBeInstanceOf

class OrderServiceTest : DescribeSpec({
    describe("OrderService") {
        val service = OrderService()
        
        describe("validateOrder") {
            it("should validate order successfully") {
                // Given
                val order = Order(
                    id = OrderId("ORD-001"),
                    customer = Customer("John Doe"),
                    items = listOf(OrderItem("item1", BigDecimal("10"))),
                    status = OrderStatus.Pending
                )
                
                // When
                val result = service.validateOrder(order)
                
                // Then
                result.shouldBeInstanceOf<Result.Success<Order>>()
                (result as Result.Success).value shouldBe order
            }
            
            it("should fail when order has no items") {
                // Given
                val order = Order(
                    id = OrderId("ORD-001"),
                    customer = Customer("John Doe"),
                    items = emptyList(),
                    status = OrderStatus.Pending
                )
                
                // When
                val result = service.validateOrder(order)
                
                // Then
                result.shouldBeInstanceOf<Result.Failure>()
            }
        }
    }
})
```

## Running Tests

```bash
# Gradle (recommended)
./gradlew test
./gradlew test --tests OrderServiceTest
./gradlew test jacocoTestReport

# Maven
mvn test
mvn test -Dtest=OrderServiceTest
mvn test jacoco:report

# View coverage report
# Gradle: build/reports/jacoco/test/html/index.html
# Maven: target/site/jacoco/index.html
```

---

## Complete Documentation

All documentation is self-contained in this directory:

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (universal)
- **[.cursorrules](./.cursorrules)** - Quick summary for Cursor IDE
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines
- **[README.md](./README.md)** - This file

**Note**: This directory is completely autonomous and works with any AI assistant (Cursor, GitHub Copilot, ChatGPT, Claude, etc.).
