# Code Quality Principles

**⚠️ CRITICAL: Read these principles before code reviews and refactoring**

## 1. Avoid Bad Trade-offs with Default Values

### ❌ **Problem**: Creating default/fallback values to mask missing data

```kotlin
// BAD: Hiding potential issues with defaults
val minimalRatePlan = roomStayPrice.ratePlans.firstOrNull() ?: RatePlan(
    ratePlanCode = null,
    commission = Commission(0, CommissionStatus.FULL),
    cancelPenalty = CancelPenalty(null, false, null),
    // ... many more defaults
)
```

### ✅ **Solution**: Fail fast with proper error handling

```kotlin
// GOOD: Explicit error handling, no hidden defaults
val ratePlanResult: Result<RatePlan> = roomStayPrice.ratePlans.firstOrNull()
    ?.let { Result.success(it) }
    ?: Result.failure("No rate plan found")
```

```kotlin
// GOOD: Explicit failure to mock and decide later
val guaranteeForCheck: Guarantee = 
    error("TODO(agt): Properly convert GuaranteeTerm to Guarantee")

// BAD: Hides with a default value trade-offs
// TODO(agt): Properly convert GuaranteeTerm to Guarantee
val guaranteeForCheck: Guarantee? = null
```

### **Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ❌ Hides problems until production
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling
- ✅ Consistent error propagation

## 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution. When you encounter similar scenarios in the codebase, use the same approach to handle them.

### Why it matters:
- ✅ Makes code predictable and easier to understand
- ✅ Reduces cognitive load when reading code
- ✅ Makes refactoring safer
- ✅ Helps identify patterns and abstractions

## 3. Extract Reusable Functions to Modules

Organize code for reuse. When you see repeated logic, extract it into a well-named function in an appropriate class or as an extension function.

### Why it matters:
- ✅ Reduces code duplication
- ✅ Makes changes easier (single point of modification)
- ✅ Improves testability
- ✅ Creates a vocabulary of reusable operations
