# AI Agent Rules for Kotlin Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on Kotlin projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
mvn test jacoco:report
# Or with Gradle
./gradlew test jacocoTestReport
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check JaCoCo reports for new/modified code (minimum 90% coverage)

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Domain-Driven Design & Test Types

**Acceptance Tests (Domain-Focused)**
- **Focus**: Business rules, domain input/output
- **Purpose**: Test business behavior, not implementation details
- **Robustness**: Should survive refactoring (high-level, stable contracts)
- **Vocabulary**: Use business/domain terminology
- **When**: Test features from user/business perspective

**Unit Tests (Technical-Focused)**
- **Focus**: Simple, isolated components (business OR technical)
- **Purpose**: Test individual methods, algorithms, patterns
- **Scope**: Lower-level than acceptance tests

#### Separation of Concerns
- **Business rules** vs **Technical rules** must be clearly separated
- Domain-Driven Design: Start with business problems and vocabulary first
- Technical code (algorithms, patterns) implements the domain model

#### The "Red" Phase Matters
1. **Design First**: The "red" phase validates design at compile time
2. **Top-Down Approach**: When appropriate, implement design by:
   - Defining method signatures with correct input/output types
   - Using business vocabulary in types and method names
   - Mocking dependencies with `TODO("Not implemented yet")` or `error("Not implemented yet")`
   - **Code must compile** even with mocks
   - Tests stay red but design is validated
3. **Design Quality Indicator**: If underlying parts are easy to implement, the design is good

#### Make It Work, Then Make It Right
- When implementing features that require refactoring:
  1. Make it work (get tests passing)
  2. Make it right (refactor without changing acceptance tests)
- **Acceptance tests should NOT change during refactoring**
- This proves the refactoring preserves business behavior

### Test Commands

```bash
# Maven
mvn test                           # Run all tests
mvn test -Dtest=ClassName          # Run specific test
mvn test jacoco:report             # Run tests with coverage
mvn verify                         # Run tests + integration tests

# Gradle
./gradlew test                     # Run all tests
./gradlew test --tests ClassName   # Run specific test
./gradlew test jacocoTestReport    # Run tests with coverage
./gradlew check                    # Run tests + checks
```

---

## Development

### Build Environment

```bash
# Maven
mvn clean install
mvn compile
mvn package

# Gradle
./gradlew clean build
./gradlew compileKotlin
./gradlew jar
```

### Code Quality Principles

#### 1. Avoid Bad Trade-offs with Default Values

**❌ Problem**: Creating default/fallback values to mask missing data

```kotlin
// BAD: Hiding potential issues with defaults
val minimalRatePlan = roomStayPrice.ratePlans.firstOrNull() ?: RatePlan(
    ratePlanCode = null,
    commission = Commission(0, CommissionStatus.FULL),
    cancelPenalty = CancelPenalty(null, false, null),
    // ... many more defaults
)
```

**✅ Solution**: Fail fast with proper error handling

```kotlin
// GOOD: Explicit error handling, no hidden defaults
val ratePlanResult: Result<RatePlan> = roomStayPrice.ratePlans.firstOrNull()
    ?.let { Result.success(it) }
    ?: Result.failure("No rate plan found")
```

```kotlin
// GOOD: Explicit failure to mock and decide later
val guaranteeForCheck: Guarantee = 
    error("TODO(agt): Properly convert GuaranteeTerm to Guarantee")

// BAD: Hides with a default value trade-offs
// TODO(agt): Properly convert GuaranteeTerm to Guarantee
val guaranteeForCheck: Guarantee? = null
```

**Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling

#### 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution

#### 3. Extract Reusable Functions to Modules

Organize for reuse

### TODO Comments

When adding todo in the code base, use this prefix: `//TODO(agt)` or use Kotlin's `TODO("message")` function

---

## Kotlin Specific Guidelines

### Kotlin-Specific Features

#### Extension Functions
✅ **Use extension functions for utility operations**

```kotlin
// Extension on String
fun String.toOrderId() = OrderId(this)

// Extension on List
fun List<OrderItem>.calculateTotal(): BigDecimal = sumOf { it.total }

// Extension on Result
fun <T> Result<T>.getOrDefault(default: T): T = when (this) {
    is Result.Success -> value
    is Result.Failure -> default
}

// Usage
val orderId = "ORD-001".toOrderId()
val total = items.calculateTotal()
```

#### Scope Functions
✅ **Use scope functions appropriately**

```kotlin
// let - for nullable transformations
val result = customer?.let { processCustomer(it) }

// run - for object configuration and computation
val order = Order(...).run {
    validateOrder(this)
    calculateTotal()
    this
}

// apply - for object configuration
val order = Order(...).apply {
    // Configure object
}

// also - for side effects
val order = createOrder(...).also {
    logger.info("Created order: ${it.id}")
}

// with - for calling multiple functions on an object
with(order) {
    validate()
    calculate()
    save()
}
```

#### When Expression
✅ **Use when instead of if-else chains**

```kotlin
fun getDiscount(customer: Customer): BigDecimal = when {
    customer.isPremium && customer.yearsActive > 5 -> BigDecimal("0.20")
    customer.isPremium -> BigDecimal("0.15")
    customer.yearsActive > 10 -> BigDecimal("0.10")
    customer.yearsActive > 5 -> BigDecimal("0.05")
    else -> BigDecimal.ZERO
}

// Exhaustive when with sealed classes
fun getStatusMessage(status: OrderStatus): String = when (status) {
    is OrderStatus.Pending -> "Order is pending"
    is OrderStatus.Confirmed -> "Order confirmed"
    is OrderStatus.Shipped -> "Shipped: ${status.trackingNumber}"
    is OrderStatus.Delivered -> "Delivered at: ${status.deliveredAt}"
}
```

### Kotlin Version
- **Minimum**: Kotlin 1.9
- **Recommended**: Kotlin 2.0+
- Use modern Kotlin features (data classes, sealed classes, coroutines, null safety, etc.)

### Required Patterns

#### Result Pattern (Railway-Oriented Programming)
✅ **Use Result pattern or Kotlin's built-in Result for error handling**

```kotlin
// Using Kotlin's built-in Result
fun validateOrder(order: Order): Result<Order> {
    return runCatching {
        validateCustomer(order)
            .getOrThrow()
            .let { validateItems(it) }
            .getOrThrow()
            .let { calculateTotal(it) }
            .getOrThrow()
    }
}

// Or custom sealed class for more control
sealed class Result<out T> {
    data class Success<T>(val value: T) : Result<T>()
    data class Failure(val error: String) : Result<Nothing>()
    
    fun <U> map(mapper: (T) -> U): Result<U> = when (this) {
        is Success -> Success(mapper(value))
        is Failure -> this
    }
    
    fun <U> flatMap(mapper: (T) -> Result<U>): Result<U> = when (this) {
        is Success -> mapper(value)
        is Failure -> this
    }
    
    companion object {
        fun <T> success(value: T): Result<T> = Success(value)
        fun <T> failure(error: String): Result<T> = Failure(error)
    }
}

// Usage
fun validateOrder(order: Order): Result<Order> {
    return validateCustomer(order)
        .flatMap { validateItems(it) }
        .flatMap { calculateTotal(it) }
}
```

#### Null Safety
✅ **Use Kotlin's null safety features**

```kotlin
// GOOD - Nullable type for nullable returns
fun findCustomer(id: String): Customer? {
    return repository.findById(id)
}

// BETTER - Result for domain operations
fun getCustomer(id: String): Result<Customer> {
    return repository.findById(id)
        ?.let { Result.success(it) }
        ?: Result.failure("Customer not found")
}

// Use safe calls and Elvis operator
fun processCustomer(id: String): String {
    val customer = findCustomer(id)
    return customer?.name ?: "Unknown"
}
```

#### Immutability with Data Classes
✅ **Use data classes for immutable data structures**

```kotlin
// Value objects
data class OrderId(val value: String) {
    init {
        require(value.isNotBlank()) { "OrderId cannot be blank" }
    }
}

data class Money(val amount: BigDecimal, val currency: Currency) {
    companion object {
        fun usd(amount: BigDecimal) = Money(amount, Currency.getInstance("USD"))
    }
}

// Domain entities
data class Order(
    val id: OrderId,
    val customer: Customer,
    val items: List<OrderItem>,
    val status: OrderStatus
) {
    init {
        require(items.isNotEmpty()) { "Order must have at least one item" }
    }
    
    fun calculateTotal(): BigDecimal = items.sumOf { it.total }
}
```

#### Sealed Classes for Domain Modeling
✅ **Use sealed classes for closed hierarchies**

```kotlin
sealed class OrderStatus {
    object Pending : OrderStatus()
    object Confirmed : OrderStatus()
    data class Shipped(val trackingNumber: String) : OrderStatus()
    data class Delivered(val deliveredAt: LocalDateTime) : OrderStatus()
}

// Pattern matching with when
fun getStatusMessage(status: OrderStatus): String = when (status) {
    is OrderStatus.Pending -> "Order is pending"
    is OrderStatus.Confirmed -> "Order confirmed"
    is OrderStatus.Shipped -> "Shipped: ${status.trackingNumber}"
    is OrderStatus.Delivered -> "Delivered at: ${status.deliveredAt}"
}
```

#### Coroutines for Async Operations
✅ **Use coroutines for asynchronous operations**

```kotlin
// Suspend functions for async operations
suspend fun fetchOrder(orderId: String): Result<Order> = coroutineScope {
    val order = async { orderRepository.findById(orderId) }
    val customer = async { customerRepository.findById(order.await().customerId) }
    
    Result.success(order.await().copy(customer = customer.await()))
}

// Flow for reactive streams
fun observeOrders(): Flow<Order> = flow {
    val orders = orderRepository.findAll()
    orders.forEach { emit(it) }
}

// Use withContext for IO operations
suspend fun saveOrder(order: Order): Result<Order> = withContext(Dispatchers.IO) {
    runCatching {
        orderRepository.save(order)
    }.fold(
        onSuccess = { Result.success(it) },
        onFailure = { Result.failure(it.message ?: "Unknown error") }
    )
}
```

#### Dependency Injection
✅ **Use constructor injection with primary constructor**

```kotlin
class OrderService(
    private val orderRepository: OrderRepository,
    private val paymentProcessor: PaymentProcessor,
    private val logger: Logger
) {
    suspend fun processOrder(order: Order): Result<Order> {
        logger.info("Processing order ${order.id}")
        return validateOrder(order)
            .flatMap { paymentProcessor.process(it) }
            .flatMap { orderRepository.save(it) }
    }
}

// With Koin or Dagger/Hilt
class OrderService @Inject constructor(
    private val orderRepository: OrderRepository,
    private val paymentProcessor: PaymentProcessor
)
```

### Kotlin Anti-Patterns to Avoid

#### ❌ Avoid Mutable State
✅ **Prefer immutable objects with val**

```kotlin
// BAD - mutable
class Order {
    var orderId: String = ""
    var totalAmount: BigDecimal = BigDecimal.ZERO
}

// GOOD - immutable with data class and val
data class Order(
    val orderId: String,
    val totalAmount: BigDecimal
)

// Use copy() for modifications
val updatedOrder = order.copy(totalAmount = newAmount)
```

#### ❌ Avoid !! (Null Assertion Operator)
✅ **Use safe calls and Elvis operator**

```kotlin
// BAD - null assertion (can throw NPE)
val name = customer!!.name

// GOOD - safe call with Elvis
val name = customer?.name ?: "Unknown"

// GOOD - early return
val customer = findCustomer(id) ?: return Result.failure("Customer not found")

// GOOD - require/check for validation
fun processCustomer(customer: Customer?) {
    requireNotNull(customer) { "Customer cannot be null" }
    // customer is now non-null
}
```

#### ❌ Avoid Blocking Calls in Coroutines
✅ **Use suspend functions and proper dispatchers**

```kotlin
// BAD - blocking call in coroutine
suspend fun fetchData(): String {
    return Thread.sleep(1000).let { "data" }  // Blocks thread!
}

// GOOD - proper suspend function
suspend fun fetchData(): String = withContext(Dispatchers.IO) {
    delay(1000)  // Non-blocking
    "data"
}

// GOOD - use async for parallel operations
suspend fun fetchOrderWithDetails(orderId: String): Order = coroutineScope {
    val order = async { fetchOrder(orderId) }
    val customer = async { fetchCustomer(order.await().customerId) }
    order.await().copy(customer = customer.await())
}
```

#### ❌ Avoid Primitive Obsession
✅ **Always wrap primitives in types with inline classes**

```kotlin
// BAD - primitive obsession
fun createOrder(customerId: String, amount: BigDecimal): Order {
    // Easy to pass parameters in wrong order
}

// GOOD - wrapped in types (value classes for zero overhead)
@JvmInline
value class CustomerId(val value: String)

data class Money(val amount: BigDecimal, val currency: Currency) {
    companion object {
        fun usd(amount: BigDecimal) = Money(amount, Currency.getInstance("USD"))
    }
}

fun createOrder(customerId: CustomerId, amount: Money): Order {
    // Type safety - can't mix up parameters
}
```

#### ❌ Use Null Safety Properly
✅ **Leverage Kotlin's null safety**

```kotlin
// GOOD - Nullable type
fun findCustomer(id: String): Customer? {
    return repository.find(id)
}

// BETTER - Result for domain operations
fun getCustomer(id: String): Result<Customer> {
    return repository.find(id)
        ?.let { Result.success(it) }
        ?: Result.failure("Customer not found")
}

// Use safe calls and Elvis operator
fun getCustomerName(id: String): String {
    return findCustomer(id)?.name ?: "Unknown"
}

// Use require/check for validation
fun processOrder(order: Order?) {
    requireNotNull(order) { "Order cannot be null" }
    check(order.items.isNotEmpty()) { "Order must have items" }
}
```

#### ❌ Prefer Result Types Over Exceptions
✅ **Use Result<T> for expected failures**

```kotlin
// BAD - exceptions for control flow
fun validateOrder(order: Order): Order {
    if (order.items.isEmpty()) {
        throw IllegalStateException("Order has no items")
    }
    return order
}

// GOOD - Result type
fun validateOrder(order: Order): Result<Order> {
    return when {
        order.items.isEmpty() -> Result.failure("Order has no items")
        else -> Result.success(order)
    }
}

// ALSO GOOD - Kotlin's built-in Result with runCatching
fun validateOrder(order: Order): kotlin.Result<Order> = runCatching {
    require(order.items.isNotEmpty()) { "Order has no items" }
    order
}
```

### Naming Conventions

#### File Naming
- **Style**: `PascalCase.kotlin`
- **Rule**: File names **must match** the class/interface name inside
- **Examples**:
  - `OrderService.kotlin` contains `class OrderService`
  - `Customer.kotlin` contains `class Customer`
  - `OrderRepository.kotlin` contains `interface OrderRepository`

#### Test File Naming
- **Format**: `ClassNameTest.kotlin`
- **Examples**:
  - `OrderService.kotlin` → `OrderServiceTest.kotlin`
  - `Customer.kotlin` → `CustomerTest.kotlin`

#### Kotlin Naming Conventions (Standard)

- **Classes, Data Classes, Interfaces**: `PascalCase`
  ```kotlin
  class OrderService
  data class Customer(val name: String, val email: String)
  interface OrderRepository
  ```

- **Functions**: `camelCase`
  ```kotlin
  fun calculateTotal(items: List<OrderItem>): BigDecimal
  fun processOrder(order: Order): Order
  ```

- **Properties/Variables**: `camelCase`
  ```kotlin
  val orderId = "12345"
  val totalAmount = BigDecimal("100.00")
  var counter = 0  // Avoid mutable vars when possible
  ```

- **Constants**: `UPPER_SNAKE_CASE` (in companion object or top-level)
  ```kotlin
  const val MAX_RETRY_COUNT = 3
  const val API_BASE_URL = "https://api.example.com"
  
  companion object {
      const val DEFAULT_TIMEOUT = 30
  }
  ```

- **Packages**: `lowercase`
  ```kotlin
  package com.example.domain
  package com.example.service
  ```

- **Extension Functions**: `camelCase`
  ```kotlin
  fun String.toOrderId() = OrderId(this)
  fun List<OrderItem>.calculateTotal() = sumOf { it.total }
  ```

### Domain Modeling

⚠️ **Ask user confirmation** when modeling domain types
- Discuss type choices (records, classes, sealed interfaces)
- Validate domain vocabulary with user
- Confirm validation and error handling approach

### Dependencies

#### Build Tools
✅ **Maven** or **Gradle** (modern choice)

**Maven** (`pom.xml`):
```xml
<properties>
    <kotlin.version>21</kotlin.version>
    <maven.compiler.source>21</maven.compiler.source>
    <maven.compiler.target>21</maven.compiler.target>
</properties>

<dependencies>
    <!-- Testing -->
    <dependency>
        <groupId>org.junit.jupiter</groupId>
        <artifactId>junit-jupiter</artifactId>
        <version>5.10.0</version>
        <scope>test</scope>
    </dependency>
    <dependency>
        <groupId>org.assertj</groupId>
        <artifactId>assertj-core</artifactId>
        <version>3.24.2</version>
        <scope>test</scope>
    </dependency>
    <dependency>
        <groupId>org.mockito</groupId>
        <artifactId>mockito-core</artifactId>
        <version>5.5.0</version>
        <scope>test</scope>
    </dependency>
</dependencies>
```

**Gradle** (`build.gradle`):
```gradle
plugins {
    id 'kotlin'
    id 'jacoco'
}

kotlin {
    toolchain {
        languageVersion = KotlinLanguageVersion.of(21)
    }
}

dependencies {
    testImplementation 'org.junit.jupiter:junit-jupiter:5.10.0'
    testImplementation 'org.assertj:assertj-core:3.24.2'
    testImplementation 'org.mockito:mockito-core:5.5.0'
}

test {
    useJUnitPlatform()
    finalizedBy jacocoTestReport
}

jacoco {
    toolVersion = "0.8.10"
}

jacocoTestReport {
    dependsOn test
    reports {
        xml.required = true
        html.required = true
    }
}
```

#### Testing Framework
✅ **JUnit 5** - Modern testing framework

```kotlin
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.DisplayName;
import static org.assertj.core.api.Assertions.*;

class OrderServiceTest {
    
    @Test
    @DisplayName("Should calculate total correctly")
    void shouldCalculateTotalCorrectly() {
        // Given
        var items = List.of(
            new OrderItem("item1", new BigDecimal("10")),
            new OrderItem("item2", new BigDecimal("20"))
        );
        
        // When
        var result = OrderService.calculateTotal(items);
        
        // Then
        assertThat(result).isEqualByComparingTo(new BigDecimal("30"));
    }
}
```

---

## Git Workflow

### Git Command Rules

1. ❌ **NEVER** use `git stash` / `git push` or its alias `psh`
2. ❌ **NEVER** use `--force` or `--force-with-lease` options
3. ⚠️ Before running a new git command not listed here, **ask to add it to the allow list**
4. ✅ Use git freely for checking differences (`git diff`, `git log`, `git status`)
5. ✅ Commit only when tests pass with assertions AND code coverage is verified

### Allowed Git Commands

```bash
git status
git diff
git log
git add <files>
git commit -m "message"
# Add more commands here as approved
```

### Commit Message Format

All commit messages must start with one of these prefixes:

- `feat(agt):` - New feature or functionality
- `fix(agt):` - Bug fixes in the codebase
- `refac(agt):` - Refactoring (preparing or finishing a feature)
- `chore(agt):` - Cleanup, removing dead code
- `doc(agt):` - Documentation changes (.md files)
- `test(agt):` - When touching test only

**Examples:**
```
feat(agt): add order validation service
fix(agt): correct tax calculation in payment processor
refac(agt): extract common validation logic to base class
chore(agt): remove deprecated API endpoints
doc(agt): update README with new API examples
```

### Commit Process

1. Run all tests (including integration tests)
2. Verify 90% code coverage minimum
3. **Ask permission to commit**
4. **Wait for user review**
5. Only commit after approval

---

## When Stuck or Uncertain

### Decision Matrix

| Situation | Action |
|-----------|--------|
| **Architecture questions** | Ask user with pros/cons example |
| **Domain modeling** | Ask user for confirmation with proposed types |
| **File location unclear** | Ask user where to create new files |
| **New git command needed** | Ask to add to allow list |
| **Design complexity** | Use top-down approach with mocks (`UnsupportedOperationException`) |
| **Test strategy unclear** | Start with acceptance tests (domain level) |
| **Coverage below 90%** | Add more test cases before committing |
| **Tests failing** | Do NOT commit - fix tests first |

### General Principle

**When in doubt, ask the user.** Collaboration is essential.

---

## Summary Checklist

Before any code modification:
- [ ] Have you read the README.md and CONTRIBUTING.md?
- [ ] Have you created and discussed a todo-list with the user?
- [ ] Have you received approval to proceed?

During development:
- [ ] Are you writing tests first (TDD)?
- [ ] Are you using Result pattern for error handling?
- [ ] Are you using records for immutable data?
- [ ] Are you wrapping primitives in types?
- [ ] Are you using Optional/Result instead of null?
- [ ] Are you using modern Kotlin features (17+)?

Before committing:
- [ ] Do all tests pass with meaningful assertions?
- [ ] Is code coverage at least 90%?
- [ ] Have you asked permission to commit?
- [ ] Have you received user approval?
- [ ] Is your commit message properly formatted with (agt) prefix?

---

**Last Updated:** 2025-12-19
**Status:** Complete - Kotlin AI Agent Guidelines
