# JavaScript AI Agents - Quick Reference

**📖 For complete documentation, see [AGENTS_RULES.md](./AGENTS_RULES.md)**

## Essential Workflow

```
Plan → Test → Develop → Test → Commit
```

1. **Plan**: Discuss with user, create todo-list, get approval
2. **Test**: Write failing tests first (TDD)
3. **Develop**: Implement with JavaScript best practices
4. **Test**: Run `npm test` (90% minimum)
5. **Commit**: Get user approval, use conventional format

## JavaScript Best Practices

### Modern ES2022+ Features
```javascript
// Async/await
async function fetchUser(id) {
  const response = await fetch(`/api/users/${id}`);
  return await response.json();
}

// Destructuring
const { name, email } = user;
const [first, ...rest] = items;

// Spread operator
const newUser = { ...user, active: true };
```

### Error Handling
```javascript
// Try-catch with async/await
async function getUser(id) {
  try {
    const user = await fetchUser(id);
    return { success: true, data: user };
  } catch (error) {
    return { success: false, error: error.message };
  }
}
```

### Functional Programming
```javascript
const adults = users
  .filter(user => user.age >= 18)
  .map(user => ({ ...user, isAdult: true }))
  .sort((a, b) => a.age - b.age);
```

### Classes
```javascript
class User {
  constructor(name, email) {
    this.id = crypto.randomUUID();
    this.name = name;
    this.email = email;
  }

  isValid() {
    return this.name && this.email.includes('@');
  }
}
```

## Testing (Jest)

```javascript
describe('User', () => {
  it('should create a valid user', () => {
    const user = new User('John', 'john@example.com');
    
    expect(user.name).toBe('John');
    expect(user.email).toBe('john@example.com');
  });

  it('should validate email', () => {
    const user = new User('John', 'invalid');
    expect(user.isValid()).toBe(false);
  });
});
```

### Run Tests
```bash
npm test                 # All tests
npm run test:coverage    # With coverage
npm run test:watch       # Watch mode
```

## Code Quality

### ❌ Don't
```javascript
// Avoid var
var x = 10;

// Avoid callback hell
getData(function(a) {
  getMoreData(a, function(b) {
    getMoreData(b, function(c) {
      // ...
    });
  });
});
```

### ✅ Do
```javascript
// Use const/let
const x = 10;
let y = 20;

// Use async/await
const a = await getData();
const b = await getMoreData(a);
const c = await getMoreData(b);
```

## Git Workflow

### Rules
- ❌ **NEVER** `git stash`
- ❌ **NEVER** `git push` or `--force`
- ✅ **ALWAYS** ask user approval before commit

### Commit Format
```
<type>(agt): <description>

feat(agt): add user authentication
fix(agt): resolve async race condition
```

## Common Commands

```bash
# Package management
npm install              # Install dependencies
npm install package      # Add package
npm update               # Update packages

# Development
npm run dev              # Dev server
npm run build            # Build for production
npm start                # Start application

# Testing
npm test                 # Run tests
npm run test:coverage    # Coverage report
npm run test:watch       # Watch mode

# Code quality
npm run lint             # ESLint
npm run format           # Prettier
```

## Quick Tips

1. **Modern JS**: Use ES2022+ features
2. **Async/Await**: For asynchronous code
3. **Const/Let**: Never use var
4. **Arrow Functions**: For concise syntax
5. **Destructuring**: For cleaner code
6. **Spread Operator**: For immutability
7. **Template Literals**: For string interpolation

## Resources

- [AGENTS_RULES.md](./AGENTS_RULES.md) - Complete rules
- [MDN Web Docs](https://developer.mozilla.org/)
- [JavaScript.info](https://javascript.info/)
