# Contributing to F# Projects

## Rules

### Architecture

Preserve the architecture but propose enhancement if needed by asking with examples by indicating pros and cons.

**Example of proposing an enhancement:**
```
Current: Services directly call external APIs
Proposed: Add Repository pattern for data access

Pros:
- Better testability (mock repositories)
- Centralized data access logic
- Easier to swap implementations

Cons:
- Additional abstraction layer
- More files to maintain
- Migration effort for existing code

Would you like me to implement this?
```

### Domain Driven Design / Acceptance Tests

As a software engineer, it is important to separate "technical" vs "business" rules. Domain Driven Design is a technique to start first with "business" problems and vocabulary but requires "technical" code (algorithm, patterns, ...) to run.

Generally **Acceptance Tests** are focused on domain as input and output (test setup, data and assertion) while **Unit Test** is focusing on testing something simple (business or technical). 

Acceptance Tests are more robust when refactoring because they are higher level and Input/Output should not change during a refactoring. At some point, implementing a feature requires a refactoring; in that case it is important to **make it work, then make it right** without changing acceptance tests.

### The "Red" Phase in TDD

The `red` part in TDD matters for compile time. Design can be validated quickly before doing implementation then refactoring. When underlying parts are easy to implement, it means that the design is good. Having business vocabulary matters in the design part.

Design matters. Sometimes, it is better to start with a **top-down approach** which consists of implementing the design by mocking underlying dependencies with a `not yet implemented` exception; the input and output of functions must be used though. The code should compile while tests will stay red but it is an important step to validate the design first to save further refactoring.

### Example: Top-Down Design

```fsharp
// Step 1: Define the high-level function with proper types
let processOrder (order: Order) : Async<Result<ProcessedOrder, OrderError>> = async {
    // Step 2: Mock dependencies with failwith
    let! validatedOrder = validateOrder order
    let! pricedOrder = calculatePricing validatedOrder
    let! savedOrder = saveOrder pricedOrder
    return Ok savedOrder
}

// Step 3: Implement each function one by one
let validateOrder (order: Order) : Async<Result<Order, OrderError>> =
    failwith "TODO(agt): Implement order validation"

let calculatePricing (order: Order) : Async<Result<Order, OrderError>> =
    failwith "TODO(agt): Implement pricing calculation"

let saveOrder (order: Order) : Async<Result<ProcessedOrder, OrderError>> =
    failwith "TODO(agt): Implement order persistence"
```

This approach:
- ✅ Validates the design at compile time
- ✅ Shows the complete workflow
- ✅ Makes it easy to implement piece by piece
- ✅ Keeps tests red until implementation is complete
