# HTML/CSS AI Agent Rules

## 🎯 Purpose

This document defines comprehensive rules for AI agents working with HTML and CSS code. These rules ensure semantic, accessible, responsive, and maintainable web interfaces.

## 📋 Table of Contents

1. [Core Principles](#core-principles)
2. [HTML Best Practices](#html-best-practices)
3. [CSS Best Practices](#css-best-practices)
4. [Responsive Design](#responsive-design)
5. [Accessibility](#accessibility)
6. [Performance](#performance)
7. [Testing Strategy](#testing-strategy)
8. [Git Workflow](#git-workflow)
9. [Code Quality](#code-quality)

## 🏗️ Core Principles

### 1. Semantic HTML First
- Use appropriate HTML5 semantic elements
- Structure content logically
- Avoid div/span soup
- Use proper heading hierarchy

### 2. Accessibility by Default
- WCAG 2.1 AA compliance minimum
- Keyboard navigation support
- Screen reader compatibility
- Proper ARIA attributes

### 3. Mobile-First Responsive
- Design for mobile first
- Progressive enhancement
- Flexible layouts (Grid/Flexbox)
- Responsive images

### 4. Performance Optimized
- Minimize HTTP requests
- Optimize assets (images, fonts)
- Reduce reflows and repaints
- Lazy loading when appropriate

### 5. Cross-Browser Compatible
- Test on major browsers
- Use feature detection
- Provide fallbacks
- Avoid browser-specific code

## 📝 HTML Best Practices

### Semantic Elements

```html
<!-- ✅ Good: Semantic HTML -->
<header>
  <nav aria-label="Main navigation">
    <ul>
      <li><a href="#home">Home</a></li>
      <li><a href="#about">About</a></li>
    </ul>
  </nav>
</header>

<main>
  <article>
    <h1>Article Title</h1>
    <p>Content...</p>
  </article>
  
  <aside>
    <h2>Related Content</h2>
  </aside>
</main>

<footer>
  <p>&copy; 2025 Company</p>
</footer>

<!-- ❌ Bad: Non-semantic divs -->
<div class="header">
  <div class="nav">
    <div class="nav-item">Home</div>
  </div>
</div>
```

### Document Structure

```html
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <meta name="description" content="Page description for SEO">
  <title>Page Title - Site Name</title>
  
  <!-- Preload critical assets -->
  <link rel="preload" href="fonts/main.woff2" as="font" type="font/woff2" crossorigin>
  
  <!-- Stylesheets -->
  <link rel="stylesheet" href="css/main.css">
</head>
<body>
  <!-- Content here -->
  
  <!-- Scripts at end of body -->
  <script src="js/main.js" defer></script>
</body>
</html>
```

### Forms and Accessibility

```html
<!-- ✅ Good: Accessible form -->
<form action="/submit" method="post">
  <div class="form-group">
    <label for="username">Username</label>
    <input 
      type="text" 
      id="username" 
      name="username" 
      required
      aria-describedby="username-help"
    >
    <span id="username-help" class="help-text">
      Enter your username (3-20 characters)
    </span>
  </div>
  
  <div class="form-group">
    <label for="email">Email</label>
    <input 
      type="email" 
      id="email" 
      name="email" 
      required
      aria-invalid="false"
    >
  </div>
  
  <button type="submit">Submit</button>
</form>
```

### Images

```html
<!-- ✅ Good: Responsive image with alt text -->
<picture>
  <source 
    media="(min-width: 1024px)" 
    srcset="images/hero-large.webp"
    type="image/webp"
  >
  <source 
    media="(min-width: 768px)" 
    srcset="images/hero-medium.webp"
    type="image/webp"
  >
  <img 
    src="images/hero-small.jpg" 
    alt="Description of the image"
    loading="lazy"
    width="800"
    height="600"
  >
</picture>

<!-- ❌ Bad: No alt text, no optimization -->
<img src="huge-image.jpg">
```

## 🎨 CSS Best Practices

### Modern Layout with Grid

```css
/* ✅ Good: CSS Grid for 2D layouts */
.container {
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
  gap: 20px;
  padding: 20px;
}

.card {
  background: white;
  border-radius: 8px;
  padding: 20px;
  box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}
```

### Flexbox for 1D Layouts

```css
/* ✅ Good: Flexbox for navigation */
.nav {
  display: flex;
  justify-content: space-between;
  align-items: center;
  padding: 1rem;
}

.nav-links {
  display: flex;
  gap: 1.5rem;
  list-style: none;
}
```

### CSS Custom Properties (Variables)

```css
/* ✅ Good: CSS Variables for theming */
:root {
  /* Colors */
  --color-primary: #2c3e50;
  --color-secondary: #3498db;
  --color-text: #333;
  --color-background: #fff;
  
  /* Spacing */
  --spacing-xs: 0.25rem;
  --spacing-sm: 0.5rem;
  --spacing-md: 1rem;
  --spacing-lg: 2rem;
  --spacing-xl: 4rem;
  
  /* Typography */
  --font-family-base: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
  --font-size-base: 1rem;
  --line-height-base: 1.6;
  
  /* Borders */
  --border-radius: 4px;
  --border-width: 1px;
}

.button {
  background: var(--color-primary);
  color: white;
  padding: var(--spacing-md) var(--spacing-lg);
  border-radius: var(--border-radius);
  font-family: var(--font-family-base);
}

/* Dark mode */
@media (prefers-color-scheme: dark) {
  :root {
    --color-text: #e0e0e0;
    --color-background: #1a1a1a;
  }
}
```

### BEM Naming Convention

```css
/* ✅ Good: BEM naming */
.card { }
.card__title { }
.card__content { }
.card--featured { }
.card__button { }
.card__button--primary { }

/* ❌ Bad: Non-semantic names */
.box1 { }
.blue-text { }
.mt-20 { }
```

### Performance-Optimized Animations

```css
/* ✅ Good: Use transform and opacity */
.card {
  transition: transform 0.2s ease, opacity 0.2s ease;
}

.card:hover {
  transform: translateY(-5px);
  opacity: 0.9;
}

/* ❌ Bad: Animating expensive properties */
.card:hover {
  width: 350px; /* Triggers layout */
  height: 450px; /* Triggers layout */
  margin-top: -10px; /* Triggers layout */
}
```

## 📱 Responsive Design

### Mobile-First Approach

```css
/* ✅ Good: Mobile first */
.container {
  width: 100%;
  padding: 1rem;
}

/* Tablet */
@media (min-width: 768px) {
  .container {
    max-width: 720px;
    margin: 0 auto;
    padding: 2rem;
  }
}

/* Desktop */
@media (min-width: 1024px) {
  .container {
    max-width: 960px;
  }
}

/* Large desktop */
@media (min-width: 1280px) {
  .container {
    max-width: 1200px;
  }
}
```

### Responsive Typography

```css
/* ✅ Good: Fluid typography */
:root {
  --font-size-base: clamp(1rem, 0.9rem + 0.5vw, 1.125rem);
  --font-size-h1: clamp(2rem, 1.5rem + 2vw, 3rem);
  --font-size-h2: clamp(1.5rem, 1.2rem + 1.5vw, 2.25rem);
}

body {
  font-size: var(--font-size-base);
}

h1 {
  font-size: var(--font-size-h1);
}

h2 {
  font-size: var(--font-size-h2);
}
```

## ♿ Accessibility

### WCAG 2.1 AA Compliance

```css
/* ✅ Good: Sufficient color contrast */
.text {
  color: #333; /* 12.6:1 contrast ratio on white */
}

.button {
  background: #2c3e50;
  color: #fff; /* 8.6:1 contrast ratio */
}

/* ✅ Good: Focus indicators */
a:focus,
button:focus,
input:focus {
  outline: 3px solid #3498db;
  outline-offset: 2px;
}

/* ❌ Bad: Removing focus outline without alternative */
*:focus {
  outline: none;
}
```

### Keyboard Navigation

```css
/* ✅ Good: Visible focus states */
.nav-link {
  position: relative;
}

.nav-link:focus {
  outline: 2px solid currentColor;
  outline-offset: 4px;
}

/* Skip link for keyboard users */
.skip-link {
  position: absolute;
  top: -40px;
  left: 0;
  background: #000;
  color: #fff;
  padding: 8px;
  text-decoration: none;
  z-index: 100;
}

.skip-link:focus {
  top: 0;
}
```

### Screen Reader Support

```html
<!-- ✅ Good: ARIA labels -->
<button aria-label="Close dialog">
  <span aria-hidden="true">&times;</span>
</button>

<nav aria-label="Main navigation">
  <!-- Navigation items -->
</nav>

<!-- Visually hidden but available to screen readers -->
<style>
.sr-only {
  position: absolute;
  width: 1px;
  height: 1px;
  padding: 0;
  margin: -1px;
  overflow: hidden;
  clip: rect(0, 0, 0, 0);
  white-space: nowrap;
  border-width: 0;
}
</style>

<span class="sr-only">Loading...</span>
```

## ⚡ Performance

### Critical CSS

```html
<!-- Inline critical CSS in <head> -->
<style>
  /* Critical above-the-fold styles */
  body { margin: 0; font-family: sans-serif; }
  .header { background: #2c3e50; padding: 1rem; }
  /* ... */
</style>

<!-- Load non-critical CSS asynchronously -->
<link rel="preload" href="css/main.css" as="style" onload="this.onload=null;this.rel='stylesheet'">
<noscript><link rel="stylesheet" href="css/main.css"></noscript>
```

### Image Optimization

```html
<!-- ✅ Good: Modern formats with fallbacks -->
<picture>
  <source srcset="image.avif" type="image/avif">
  <source srcset="image.webp" type="image/webp">
  <img src="image.jpg" alt="Description" loading="lazy">
</picture>
```

### CSS Optimization

```css
/* ✅ Good: Efficient selectors */
.card { }
.card-title { }

/* ❌ Bad: Overly specific selectors */
div.container > div.row > div.col > div.card > h2.title { }

/* ✅ Good: Avoid expensive properties */
.element {
  will-change: transform; /* Only when needed */
  transform: translateZ(0); /* Create layer */
}
```

## 🧪 Testing Strategy

### HTML Validation
- Use W3C Markup Validation Service
- HTMLHint for linting
- Check semantic structure

### CSS Validation
- Use W3C CSS Validation Service
- Stylelint for linting
- Check for unused CSS

### Accessibility Testing
- axe DevTools
- WAVE browser extension
- Lighthouse accessibility audit
- Keyboard navigation testing
- Screen reader testing (NVDA, JAWS, VoiceOver)

### Cross-Browser Testing
- Chrome, Firefox, Safari, Edge
- Mobile browsers (iOS Safari, Chrome Mobile)
- BrowserStack or similar service

### Performance Testing
- Lighthouse performance audit
- PageSpeed Insights
- WebPageTest
- Check Core Web Vitals (LCP, FID, CLS)

## 🔄 Git Workflow

### Commit Message Format

```
type(html/css): brief description

Detailed explanation of changes

Examples:
- feat(html): add semantic article structure
- fix(css): correct grid layout on mobile
- style(css): improve button hover states
- refactor(html): simplify navigation markup
- perf(css): optimize animation performance
- a11y(html): add ARIA labels to form inputs
```

### Branch Strategy
- `main`: Production-ready code
- `develop`: Integration branch
- `feature/description`: New features
- `fix/description`: Bug fixes
- `refactor/description`: Code refactoring

### Before Committing
1. Validate HTML (W3C Validator)
2. Lint CSS (Stylelint)
3. Test accessibility (axe DevTools)
4. Test responsive design
5. Cross-browser check
6. Performance audit

## 📊 Code Quality Principles

### 1. Maintainability
- Consistent naming conventions
- Logical file organization
- Comments for complex logic
- Modular CSS architecture

### 2. Reusability
- Component-based approach
- Utility classes for common patterns
- CSS custom properties for theming

### 3. Scalability
- BEM or similar methodology
- CSS architecture (ITCSS, SMACSS)
- Avoid deep nesting (max 3 levels)

### 4. Documentation
- Comment complex selectors
- Document color palette
- Explain layout decisions
- Maintain style guide

## 🚫 Anti-Patterns to Avoid

### HTML
- ❌ Non-semantic div/span soup
- ❌ Inline styles
- ❌ Missing alt text on images
- ❌ Improper heading hierarchy
- ❌ Forms without labels
- ❌ Tables for layout

### CSS
- ❌ !important overuse
- ❌ Magic numbers without explanation
- ❌ Fixed pixel widths
- ❌ Deep nesting (> 3 levels)
- ❌ Non-descriptive class names
- ❌ Inline styles in HTML
- ❌ Browser-specific prefixes without autoprefixer

## 📚 Resources

### Documentation
- [MDN Web Docs](https://developer.mozilla.org/)
- [W3C Standards](https://www.w3.org/standards/)
- [WCAG Guidelines](https://www.w3.org/WAI/WCAG21/quickref/)

### Tools
- [Can I Use](https://caniuse.com/) - Browser support
- [CSS-Tricks](https://css-tricks.com/) - CSS techniques
- [A11y Project](https://www.a11yproject.com/) - Accessibility

### Validation
- [W3C HTML Validator](https://validator.w3.org/)
- [W3C CSS Validator](https://jigsaw.w3.org/css-validator/)
- [axe DevTools](https://www.deque.com/axe/devtools/)

---

**Version**: 1.0.0  
**Last Updated**: 2025-12-22  
**Maintained by**: AI Agents Configuration Team
