# Go Project Template

This is a Go project configured for AI-assisted development with TDD and DDD principles.

## 📋 Quick Start

### Prerequisites

- Go 1.21+ installed
- Git configured
- (Optional) golangci-lint for linting

### Setup

```bash
# Initialize Go module
go mod init github.com/yourusername/yourproject

# Install dependencies
go mod tidy

# Run tests
go test ./...

# Build
go build -o myapp ./cmd/myapp
```

## 🏗️ Project Structure

```
myproject/
├── cmd/
│   └── myapp/
│       └── main.go           # Application entry point
├── internal/
│   ├── domain/               # Domain models and business logic
│   │   ├── user.go
│   │   └── user_test.go
│   ├── repository/           # Data access layer
│   │   └── postgres/
│   │       ├── user_repo.go
│   │       └── user_repo_test.go
│   ├── service/              # Application services
│   │   ├── user_service.go
│   │   └── user_service_test.go
│   └── handler/              # HTTP handlers
│       ├── user_handler.go
│       └── user_handler_test.go
├── pkg/                      # Public libraries (reusable)
│   └── api/
├── test/                     # Additional test files
│   └── integration/
├── scripts/                  # Build and deployment scripts
├── docs/                     # Documentation
├── go.mod                    # Go module definition
├── go.sum                    # Dependency checksums
├── .gitignore
├── AGENTS_RULES.md           # AI agent rules (READ THIS FIRST)
├── AGENTS.md                 # Quick reference
├── CODE_QUALITY_PRINCIPLES.md
├── CONTRIBUTING.md
└── README.md                 # This file
```

## 🧪 Testing

### Run Tests

```bash
# Run all tests
go test ./...

# Run tests with coverage
go test ./... -cover

# Generate coverage report
go test ./... -coverprofile=coverage.out
go tool cover -html=coverage.out -o coverage.html

# Run specific test
go test -run TestUserValidation ./internal/domain

# Run with race detector
go test -race ./...

# Run benchmarks
go test -bench=. ./...
```

### Test Organization

```go
// Example: Table-driven test
func TestValidateUser(t *testing.T) {
    tests := []struct {
        name    string
        user    User
        wantErr bool
    }{
        {
            name:    "valid user",
            user:    User{Name: "John", Email: "john@example.com"},
            wantErr: false,
        },
        {
            name:    "empty name",
            user:    User{Name: "", Email: "john@example.com"},
            wantErr: true,
        },
        {
            name:    "invalid email",
            user:    User{Name: "John", Email: "invalid"},
            wantErr: true,
        },
    }
    
    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            err := ValidateUser(tt.user)
            if (err != nil) != tt.wantErr {
                t.Errorf("ValidateUser() error = %v, wantErr %v", err, tt.wantErr)
            }
        })
    }
}
```

## 🔧 Development

### Build

```bash
# Build for current platform
go build -o myapp ./cmd/myapp

# Build for specific platform
GOOS=linux GOARCH=amd64 go build -o myapp-linux ./cmd/myapp
GOOS=windows GOARCH=amd64 go build -o myapp.exe ./cmd/myapp

# Build with version info
go build -ldflags "-X main.version=1.0.0" -o myapp ./cmd/myapp
```

### Run

```bash
# Run directly
go run ./cmd/myapp

# Run with arguments
go run ./cmd/myapp --config=config.yaml

# Run with environment variables
PORT=8080 go run ./cmd/myapp
```

### Code Quality

```bash
# Format code
go fmt ./...
gofmt -w .

# Lint code
golangci-lint run

# Static analysis
go vet ./...

# Check for common mistakes
staticcheck ./...
```

## 📦 Dependencies

### Add Dependency

```bash
# Add a dependency
go get github.com/pkg/errors

# Add specific version
go get github.com/pkg/errors@v0.9.1

# Update dependency
go get -u github.com/pkg/errors

# Update all dependencies
go get -u ./...
```

### Manage Dependencies

```bash
# Clean up unused dependencies
go mod tidy

# Vendor dependencies
go mod vendor

# Verify dependencies
go mod verify

# Show dependency graph
go mod graph
```

## 🎯 Example: User Domain

### Domain Model

```go
// internal/domain/user.go
package domain

import (
    "errors"
    "regexp"
)

type User struct {
    ID    string
    Name  string
    Email string
}

func NewUser(name, email string) (*User, error) {
    if name == "" {
        return nil, errors.New("name is required")
    }
    if !isValidEmail(email) {
        return nil, errors.New("invalid email")
    }
    
    return &User{
        ID:    generateID(),
        Name:  name,
        Email: email,
    }, nil
}

func (u *User) UpdateEmail(email string) error {
    if !isValidEmail(email) {
        return errors.New("invalid email")
    }
    u.Email = email
    return nil
}

func isValidEmail(email string) bool {
    re := regexp.MustCompile(`^[a-z0-9._%+\-]+@[a-z0-9.\-]+\.[a-z]{2,}$`)
    return re.MatchString(email)
}
```

### Repository Interface

```go
// internal/domain/user_repository.go
package domain

import "context"

type UserRepository interface {
    FindByID(ctx context.Context, id string) (*User, error)
    FindByEmail(ctx context.Context, email string) (*User, error)
    Save(ctx context.Context, user *User) error
    Delete(ctx context.Context, id string) error
}
```

### Service

```go
// internal/service/user_service.go
package service

import (
    "context"
    "fmt"
    
    "yourproject/internal/domain"
)

type UserService struct {
    repo domain.UserRepository
}

func NewUserService(repo domain.UserRepository) *UserService {
    return &UserService{repo: repo}
}

func (s *UserService) RegisterUser(ctx context.Context, name, email string) (*domain.User, error) {
    // Check if user exists
    existing, err := s.repo.FindByEmail(ctx, email)
    if err == nil && existing != nil {
        return nil, fmt.Errorf("user with email %s already exists", email)
    }
    
    // Create new user
    user, err := domain.NewUser(name, email)
    if err != nil {
        return nil, fmt.Errorf("invalid user data: %w", err)
    }
    
    // Save user
    if err := s.repo.Save(ctx, user); err != nil {
        return nil, fmt.Errorf("failed to save user: %w", err)
    }
    
    return user, nil
}
```

## 🚀 Deployment

### Docker

```dockerfile
# Dockerfile
FROM golang:1.21-alpine AS builder

WORKDIR /app
COPY go.mod go.sum ./
RUN go mod download

COPY . .
RUN CGO_ENABLED=0 GOOS=linux go build -o myapp ./cmd/myapp

FROM alpine:latest
RUN apk --no-cache add ca-certificates
WORKDIR /root/
COPY --from=builder /app/myapp .

EXPOSE 8080
CMD ["./myapp"]
```

```bash
# Build and run
docker build -t myapp .
docker run -p 8080:8080 myapp
```

### Makefile

```makefile
# Makefile
.PHONY: build test clean run

build:
	go build -o bin/myapp ./cmd/myapp

test:
	go test ./... -cover

test-coverage:
	go test ./... -coverprofile=coverage.out
	go tool cover -html=coverage.out -o coverage.html

clean:
	rm -rf bin/
	rm -f coverage.out coverage.html

run:
	go run ./cmd/myapp

lint:
	golangci-lint run

fmt:
	go fmt ./...
	gofmt -w .
```

## 📚 Documentation

- **[AGENTS_RULES.md](./AGENTS_RULES.md)** - Complete AI agent rules (READ THIS FIRST)
- **[AGENTS.md](./AGENTS.md)** - Quick reference guide
- **[CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md)** - Code quality principles
- **[CONTRIBUTING.md](./CONTRIBUTING.md)** - Contribution guidelines

## 🔗 Resources

### Official
- [Go Documentation](https://go.dev/doc/)
- [Effective Go](https://go.dev/doc/effective_go)
- [Go Code Review Comments](https://github.com/golang/go/wiki/CodeReviewComments)

### Community
- [Uber Go Style Guide](https://github.com/uber-go/guide/blob/master/style.md)
- [Standard Go Project Layout](https://github.com/golang-standards/project-layout)
- [Go Proverbs](https://go-proverbs.github.io/)

### Tools
- [golangci-lint](https://golangci-lint.run/) - Fast linters runner
- [staticcheck](https://staticcheck.io/) - Static analysis
- [testify](https://github.com/stretchr/testify) - Testing toolkit

## 🤝 Contributing

See [CONTRIBUTING.md](./CONTRIBUTING.md) for detailed guidelines.

Quick summary:
1. Read [AGENTS_RULES.md](./AGENTS_RULES.md) completely
2. Follow TDD: Write tests first
3. Maintain 90% code coverage
4. Use conventional commit format
5. Get approval before committing

## 📄 License

[Your License Here]
