# Contributing to Go Projects

## Rules

### Architecture

Preserve the architecture but propose enhancement if needed by asking with examples by indicating pros and cons.

**Example of proposing an enhancement:**
```
Current: Services directly call external APIs
Proposed: Add Repository pattern for data access

Pros:
- Better testability (mock repositories)
- Centralized data access logic
- Easier to swap implementations

Cons:
- Additional abstraction layer
- More files to maintain
- Migration effort for existing code

Would you like me to implement this?
```

### Domain Driven Design / Acceptance Tests

As a software engineer, it is important to separate "technical" vs "business" rules. Domain Driven Design is a technique to start first with "business" problems and vocabulary but requires "technical" code (algorithm, patterns, ...) to run.

Generally **Acceptance Tests** are focused on domain as input and output (test setup, data and assertion) while **Unit Test** is focusing on testing something simple (business or technical). 

Acceptance Tests are more robust when refactoring because they are higher level and Input/Output should not change during a refactoring. At some point, implementing a feature requires a refactoring; in that case it is important to **make it work, then make it right** without changing acceptance tests.

### The "Red" Phase in TDD

The `red` part in TDD matters for compile time. Design can be validated quickly before doing implementation then refactoring. When underlying parts are easy to implement, it means that the design is good. Having business vocabulary matters in the design part.

Design matters. Sometimes, it is better to start with a **top-down approach** which consists of implementing the design by mocking underlying dependencies with a `not yet implemented` error; the input and output of functions must be used though. The code should compile while tests will stay red but it is an important step to validate the design first to save further refactoring.

### Example: Top-Down Design in Go

```go
// Step 1: Define the high-level function with proper types
func ProcessOrder(ctx context.Context, order *Order) (*ProcessedOrder, error) {
    // Step 2: Mock dependencies with errors
    validatedOrder, err := validateOrder(ctx, order)
    if err != nil {
        return nil, fmt.Errorf("validation failed: %w", err)
    }
    
    pricedOrder, err := calculatePricing(ctx, validatedOrder)
    if err != nil {
        return nil, fmt.Errorf("pricing failed: %w", err)
    }
    
    savedOrder, err := saveOrder(ctx, pricedOrder)
    if err != nil {
        return nil, fmt.Errorf("save failed: %w", err)
    }
    
    return savedOrder, nil
}

// Step 3: Implement each function one by one
func validateOrder(ctx context.Context, order *Order) (*Order, error) {
    // TODO(agt): Implement order validation
    return nil, errors.New("not implemented: order validation")
}

func calculatePricing(ctx context.Context, order *Order) (*Order, error) {
    // TODO(agt): Implement pricing calculation
    return nil, errors.New("not implemented: pricing calculation")
}

func saveOrder(ctx context.Context, order *Order) (*ProcessedOrder, error) {
    // TODO(agt): Implement order persistence
    return nil, errors.New("not implemented: order persistence")
}
```

This approach:
- ✅ Validates the design at compile time
- ✅ Shows the complete workflow
- ✅ Makes it easy to implement piece by piece
- ✅ Keeps tests red until implementation is complete

## Test-Driven Development Workflow

### 1. Red Phase: Write a Failing Test

```go
func TestUserService_RegisterUser(t *testing.T) {
    // Arrange
    repo := &MockUserRepository{}
    service := NewUserService(repo)
    
    // Act
    user, err := service.RegisterUser(context.Background(), "John", "john@example.com")
    
    // Assert
    if err != nil {
        t.Fatalf("unexpected error: %v", err)
    }
    if user.Name != "John" {
        t.Errorf("expected name John, got %s", user.Name)
    }
    if user.Email != "john@example.com" {
        t.Errorf("expected email john@example.com, got %s", user.Email)
    }
}
```

### 2. Green Phase: Make it Pass

```go
func (s *UserService) RegisterUser(ctx context.Context, name, email string) (*User, error) {
    user := &User{
        ID:    generateID(),
        Name:  name,
        Email: email,
    }
    
    if err := s.repo.Save(ctx, user); err != nil {
        return nil, fmt.Errorf("failed to save user: %w", err)
    }
    
    return user, nil
}
```

### 3. Refactor Phase: Improve the Code

```go
// Extract validation
func (s *UserService) RegisterUser(ctx context.Context, name, email string) (*User, error) {
    if err := validateUserInput(name, email); err != nil {
        return nil, err
    }
    
    user := &User{
        ID:    generateID(),
        Name:  name,
        Email: email,
    }
    
    if err := s.repo.Save(ctx, user); err != nil {
        return nil, fmt.Errorf("failed to save user: %w", err)
    }
    
    return user, nil
}

func validateUserInput(name, email string) error {
    if name == "" {
        return errors.New("name is required")
    }
    if !isValidEmail(email) {
        return errors.New("invalid email format")
    }
    return nil
}
```

## Code Organization

### Package Structure

```
myproject/
├── cmd/                    # Application entry points
│   └── myapp/
│       └── main.go
├── internal/               # Private application code
│   ├── domain/            # Domain models and business logic
│   │   ├── user.go
│   │   └── user_test.go
│   ├── repository/        # Data access layer
│   │   ├── user_repository.go
│   │   └── postgres/
│   │       ├── user_repo.go
│   │       └── user_repo_test.go
│   ├── service/           # Application services
│   │   ├── user_service.go
│   │   └── user_service_test.go
│   └── handler/           # HTTP handlers
│       ├── user_handler.go
│       └── user_handler_test.go
└── pkg/                    # Public libraries
    └── validation/
        ├── email.go
        └── email_test.go
```

### File Naming Conventions

- `user.go` - Domain model
- `user_test.go` - Tests for user.go
- `user_repository.go` - Interface definition
- `user_repo.go` - Repository implementation
- `user_service.go` - Service implementation
- `user_handler.go` - HTTP handler

## Testing Guidelines

### Table-Driven Tests

```go
func TestValidateEmail(t *testing.T) {
    tests := []struct {
        name    string
        email   string
        wantErr bool
    }{
        {"valid email", "test@example.com", false},
        {"empty email", "", true},
        {"no @", "testexample.com", true},
        {"no domain", "test@", true},
        {"no TLD", "test@example", true},
    }
    
    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            err := ValidateEmail(tt.email)
            if (err != nil) != tt.wantErr {
                t.Errorf("ValidateEmail() error = %v, wantErr %v", err, tt.wantErr)
            }
        })
    }
}
```

### Mocking with Interfaces

```go
// Define interface
type UserRepository interface {
    FindByID(ctx context.Context, id string) (*User, error)
    Save(ctx context.Context, user *User) error
}

// Mock implementation for testing
type MockUserRepository struct {
    SaveFunc    func(ctx context.Context, user *User) error
    FindByIDFunc func(ctx context.Context, id string) (*User, error)
}

func (m *MockUserRepository) Save(ctx context.Context, user *User) error {
    if m.SaveFunc != nil {
        return m.SaveFunc(ctx, user)
    }
    return nil
}

func (m *MockUserRepository) FindByID(ctx context.Context, id string) (*User, error) {
    if m.FindByIDFunc != nil {
        return m.FindByIDFunc(ctx, id)
    }
    return nil, errors.New("not found")
}

// Usage in tests
func TestUserService_RegisterUser(t *testing.T) {
    repo := &MockUserRepository{
        SaveFunc: func(ctx context.Context, user *User) error {
            // Verify user data
            if user.Name == "" {
                return errors.New("name is required")
            }
            return nil
        },
    }
    
    service := NewUserService(repo)
    // ... test code
}
```

### Test Coverage

Aim for **90% code coverage** minimum:

```bash
# Run tests with coverage
go test ./... -coverprofile=coverage.out

# View coverage report
go tool cover -html=coverage.out

# Check coverage percentage
go test ./... -cover
```

## Git Workflow

### Commit Message Format

Use conventional commits:

```
<type>(agt): <description>

[optional body]

[optional footer]
```

**Types:**
- `feat(agt):` - New feature
- `fix(agt):` - Bug fix
- `refac(agt):` - Refactoring (no functional changes)
- `chore(agt):` - Maintenance tasks
- `doc(agt):` - Documentation
- `test(agt):` - Tests only

**Examples:**
```
feat(agt): add user authentication with JWT

Implement JWT-based authentication for user login.
Includes token generation, validation, and refresh logic.

Closes #123
```

```
fix(agt): resolve race condition in order processing

Use mutex to protect shared state in concurrent order processing.
Adds race detector tests to prevent regression.
```

```
refac(agt): extract payment logic into separate package

Move payment-related code from service to dedicated payment package.
No functional changes, improves code organization.
```

### Git Rules

**⚠️ CRITICAL:**
- ❌ **NEVER** use `git stash`
- ❌ **NEVER** use `git push` or `git push --force`
- ✅ **ALWAYS** ask user approval before committing

### Commit Workflow

1. **Run tests** and verify coverage
2. **Review changes** with user
3. **Ask for commit approval**
4. **Create commit** with proper message
5. **Confirm** commit was successful

```bash
# Stage changes
git add .

# Commit with message
git commit -m "feat(agt): implement user registration"

# Verify commit
git log -1
```

## Code Review Checklist

Before requesting review:

- [ ] All tests pass
- [ ] Code coverage is at least 90%
- [ ] No linter warnings (`golangci-lint run`)
- [ ] Code is formatted (`go fmt ./...`)
- [ ] Error handling is explicit
- [ ] Interfaces are used appropriately
- [ ] Context is passed correctly
- [ ] No race conditions (tested with `-race`)
- [ ] Documentation is updated
- [ ] Commit message follows format

## Best Practices Summary

1. **TDD**: Write tests first (Red-Green-Refactor)
2. **DDD**: Separate business and technical concerns
3. **Top-Down**: Design with mocks, implement incrementally
4. **Interfaces**: Accept interfaces, return structs
5. **Errors**: Check every error, wrap with context
6. **Testing**: Table-driven tests, 90% coverage
7. **Git**: Conventional commits, user approval required

## Resources

- [AGENTS_RULES.md](./AGENTS_RULES.md) - Complete AI agent rules
- [CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md) - Quality principles
- [Effective Go](https://go.dev/doc/effective_go) - Official guide
- [Uber Go Style Guide](https://github.com/uber-go/guide/blob/master/style.md)

---

**Remember**: Read [AGENTS_RULES.md](./AGENTS_RULES.md) completely before starting any work.
