# Code Quality Principles for Go

**⚠️ CRITICAL: Read these principles before code reviews and refactoring**

## 1. Avoid Bad Trade-offs with Default Values

### ❌ **Problem**: Creating default/fallback values to mask missing data

```go
// BAD: Hiding potential issues with defaults
func GetRatePlan(roomStay *RoomStay) *RatePlan {
    if len(roomStay.RatePlans) == 0 {
        // Creating fake data with many defaults
        return &RatePlan{
            Code:           "",
            Commission:     0,
            CancelDeadline: nil,
            IsNonRefundable: false,
            // ... many more defaults
        }
    }
    return roomStay.RatePlans[0]
}
```

### ✅ **Solution**: Fail fast with proper error handling

```go
// GOOD: Explicit error handling, no hidden defaults
func GetRatePlan(roomStay *RoomStay) (*RatePlan, error) {
    if len(roomStay.RatePlans) == 0 {
        return nil, errors.New("no rate plans available")
    }
    return roomStay.RatePlans[0], nil
}

// Usage
ratePlan, err := GetRatePlan(roomStay)
if err != nil {
    return fmt.Errorf("failed to get rate plan: %w", err)
}
```

```go
// GOOD: Explicit failure to mock and decide later
// Tests may fail but it is a baby step to avoid adding too much complexity at first
func ConvertGuarantee(term *GuaranteeTerm) (*Guarantee, error) {
    // TODO(agt): Properly convert GuaranteeTerm to Guarantee
    return nil, errors.New("not implemented: guarantee conversion")
}

// BAD: Hides with a default value trade-offs
func ConvertGuarantee(term *GuaranteeTerm) *Guarantee {
    // TODO(agt): Properly convert GuaranteeTerm to Guarantee
    return nil // Silent failure
}
```

### **Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ❌ Hides problems until production
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling
- ✅ Consistent error propagation

## 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution. When you encounter similar scenarios in the codebase, use the same approach to handle them.

### ❌ **Problem**: Inconsistent error handling

```go
// BAD: Mixing error handling strategies
func ProcessOrder(order *Order) interface{} {
    if !order.IsValid() {
        return errors.New("invalid order") // Returns error
    }
    result := &ProcessResult{Success: true}
    return result // Returns result
}
```

### ✅ **Solution**: Consistent return types

```go
// GOOD: Consistent error handling
func ProcessOrder(order *Order) (*ProcessResult, error) {
    if !order.IsValid() {
        return nil, errors.New("invalid order")
    }
    result := &ProcessResult{Success: true}
    return result, nil
}
```

### Why it matters:
- ✅ Makes code predictable and easier to understand
- ✅ Reduces cognitive load when reading code
- ✅ Makes refactoring safer
- ✅ Helps identify patterns and abstractions

## 3. Extract Reusable Functions to Packages

Organize code for reuse. When you see repeated logic, extract it into a well-named function in an appropriate package.

### ❌ **Problem**: Repeated validation logic

```go
// BAD: Duplicated validation
func CreateUser(name, email string) (*User, error) {
    if name == "" {
        return nil, errors.New("name is required")
    }
    emailRegex := regexp.MustCompile(`^[a-z0-9._%+\-]+@[a-z0-9.\-]+\.[a-z]{2,}$`)
    if !emailRegex.MatchString(email) {
        return nil, errors.New("invalid email")
    }
    // ...
}

func UpdateUser(user *User, email string) error {
    // Same validation duplicated
    emailRegex := regexp.MustCompile(`^[a-z0-9._%+\-]+@[a-z0-9.\-]+\.[a-z]{2,}$`)
    if !emailRegex.MatchString(email) {
        return errors.New("invalid email")
    }
    // ...
}
```

### ✅ **Solution**: Extract to reusable function

```go
// GOOD: Extracted validation
package validation

var emailRegex = regexp.MustCompile(`^[a-z0-9._%+\-]+@[a-z0-9.\-]+\.[a-z]{2,}$`)

func ValidateEmail(email string) error {
    if !emailRegex.MatchString(email) {
        return errors.New("invalid email format")
    }
    return nil
}

func ValidateName(name string) error {
    if name == "" {
        return errors.New("name is required")
    }
    return nil
}

// Usage
func CreateUser(name, email string) (*User, error) {
    if err := validation.ValidateName(name); err != nil {
        return nil, err
    }
    if err := validation.ValidateEmail(email); err != nil {
        return nil, err
    }
    // ...
}
```

### Why it matters:
- ✅ Reduces code duplication
- ✅ Makes changes easier (single point of modification)
- ✅ Improves testability
- ✅ Creates a vocabulary of reusable operations

## 4. Go-Specific Quality Principles

### Use Interfaces Appropriately

```go
// GOOD: Accept interfaces, return structs
func ProcessUser(repo UserRepository, userID string) (*User, error) {
    return repo.FindByID(userID)
}

// BAD: Returning interface makes testing harder
func ProcessUser(repo UserRepository, userID string) (UserRepository, error) {
    // ...
}
```

### Design for Zero Values

```go
// GOOD: Zero value is useful
type Config struct {
    Timeout time.Duration // 0 is a valid timeout
    MaxRetries int        // 0 means no retries
}

// GOOD: Use pointer for truly optional fields
type User struct {
    Name  string
    Email *string // nil means no email provided
}
```

### Use Context Properly

```go
// GOOD: Context as first parameter
func FetchUser(ctx context.Context, userID string) (*User, error) {
    select {
    case <-ctx.Done():
        return nil, ctx.Err()
    default:
        return repo.FindByID(userID)
    }
}

// BAD: Context stored in struct
type Service struct {
    ctx context.Context // Don't do this
}
```

### Handle Errors Explicitly

```go
// GOOD: Check every error
file, err := os.Open("config.yaml")
if err != nil {
    return fmt.Errorf("failed to open config: %w", err)
}
defer file.Close()

// BAD: Ignoring errors
file, _ := os.Open("config.yaml")
defer file.Close()
```

## 5. Testing Quality Principles

### Use Table-Driven Tests

```go
// GOOD: Table-driven test
func TestValidateEmail(t *testing.T) {
    tests := []struct {
        name    string
        email   string
        wantErr bool
    }{
        {"valid", "test@example.com", false},
        {"empty", "", true},
        {"no @", "testexample.com", true},
        {"no domain", "test@", true},
    }
    
    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            err := ValidateEmail(tt.email)
            if (err != nil) != tt.wantErr {
                t.Errorf("got error %v, wantErr %v", err, tt.wantErr)
            }
        })
    }
}
```

### Test Behavior, Not Implementation

```go
// GOOD: Testing behavior
func TestUserService_RegisterUser(t *testing.T) {
    repo := &MockUserRepository{}
    service := NewUserService(repo)
    
    user, err := service.RegisterUser(context.Background(), "John", "john@example.com")
    
    if err != nil {
        t.Fatalf("unexpected error: %v", err)
    }
    if user.Name != "John" {
        t.Errorf("expected name John, got %s", user.Name)
    }
    if !repo.SaveCalled {
        t.Error("expected Save to be called")
    }
}

// BAD: Testing implementation details
func TestUserService_RegisterUser(t *testing.T) {
    // Testing internal method calls instead of behavior
    // ...
}
```

## 6. Concurrency Quality Principles

### Use Channels for Communication

```go
// GOOD: Channels for goroutine communication
func ProcessItems(items []Item) []Result {
    results := make(chan Result, len(items))
    
    for _, item := range items {
        go func(i Item) {
            results <- process(i)
        }(item)
    }
    
    var collected []Result
    for i := 0; i < len(items); i++ {
        collected = append(collected, <-results)
    }
    return collected
}
```

### Avoid Shared Mutable State

```go
// BAD: Shared mutable state
var counter int
func increment() {
    counter++ // Race condition!
}

// GOOD: Use mutex or channels
type Counter struct {
    mu    sync.Mutex
    value int
}

func (c *Counter) Increment() {
    c.mu.Lock()
    defer c.mu.Unlock()
    c.value++
}

// BETTER: Use atomic operations for simple cases
var counter int64
func increment() {
    atomic.AddInt64(&counter, 1)
}
```

## Summary

1. **Fail Fast**: No default values to hide errors
2. **Be Consistent**: Same problem = same solution
3. **Extract & Reuse**: DRY principle, create reusable functions
4. **Use Interfaces Wisely**: Accept interfaces, return structs
5. **Handle Errors**: Check every error, wrap with context
6. **Test Behavior**: Table-driven tests, test what matters
7. **Safe Concurrency**: Channels, mutexes, atomic operations

**Remember**: These principles make code more maintainable, testable, and reliable.
