# AI Agent Rules for Go Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on Go projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
go test ./... -v -cover -coverprofile=coverage.out
go tool cover -html=coverage.out -o coverage.html
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check coverage report for new/modified code (minimum 90% coverage)

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Domain-Driven Design & Test Types

**Acceptance Tests (Domain-Focused)**
- **Focus**: Business rules, domain input/output
- **Purpose**: Test business behavior, not implementation details
- **Robustness**: Should survive refactoring (high-level, stable contracts)
- **Vocabulary**: Use business/domain terminology
- **When**: Test features from user/business perspective

**Unit Tests (Technical-Focused)**
- **Focus**: Simple, isolated components (business OR technical)
- **Purpose**: Test individual functions, algorithms, patterns
- **Scope**: Lower-level than acceptance tests

#### Separation of Concerns
- **Business rules** vs **Technical rules** must be clearly separated
- Domain-Driven Design: Start with business problems and vocabulary first

#### Test Organization
- Use table-driven tests for multiple test cases
- Use subtests with `t.Run()` for better organization
- Use test helpers to reduce duplication
- Mock external dependencies with interfaces

---

## Development

### Go-Specific Best Practices

#### 1. Error Handling
- **Always check errors** - Never ignore returned errors
- **Wrap errors with context** using `fmt.Errorf("context: %w", err)`
- **Return errors, don't panic** - Panics are for truly exceptional situations
- **Use custom error types** when appropriate

```go
// Good: Explicit error handling
result, err := doSomething()
if err != nil {
    return fmt.Errorf("failed to do something: %w", err)
}

// Bad: Ignoring errors
result, _ := doSomething()
```

#### 2. Interfaces
- **Accept interfaces, return structs**
- **Keep interfaces small** - Prefer single-method interfaces
- **Define interfaces at point of use**, not with implementation
- **Use standard library interfaces** when possible (io.Reader, io.Writer, etc.)

```go
// Good: Small, focused interface
type UserRepository interface {
    FindByID(id string) (*User, error)
}

// Bad: Large, monolithic interface
type Repository interface {
    FindByID(id string) (*User, error)
    FindAll() ([]*User, error)
    Save(user *User) error
    Delete(id string) error
    // ... many more methods
}
```

#### 3. Structs and Methods
- **Use pointer receivers** for methods that modify the receiver
- **Use value receivers** for methods that don't modify the receiver
- **Be consistent** with receiver types within a type
- **Use constructor functions** (NewXxx) for complex initialization

```go
type User struct {
    ID    string
    Name  string
    Email string
}

// Constructor
func NewUser(name, email string) (*User, error) {
    if name == "" || email == "" {
        return nil, errors.New("name and email are required")
    
    return &User{
        ID:    generateID(),
        Name:  name,
        Email: email,
    }, nil
}

// Pointer receiver (modifies receiver)
func (u *User) UpdateEmail(email string) error {
    if email == "" {
        return errors.New("email cannot be empty")
    }
    u.Email = email
    return nil
}

// Value receiver (doesn't modify receiver)
func (u User) String() string {
    return fmt.Sprintf("%s <%s>", u.Name, u.Email)
}
```

#### 4. Concurrency
- **Use goroutines for concurrent operations**
- **Use channels for communication** between goroutines
- **Use sync.WaitGroup** for waiting on multiple goroutines
- **Use context.Context** for cancellation and timeouts
- **Avoid shared mutable state** - Use channels or mutexes

```go
// Good: Using context and channels
func ProcessItems(ctx context.Context, items []Item) error {
    results := make(chan Result, len(items))
    errors := make(chan error, len(items))
    
    for _, item := range items {
        go func(i Item) {
            select {
            case <-ctx.Done():
                errors <- ctx.Err()
                return
            default:
                result, err := process(i)
                if err != nil {
                    errors <- err
                    return
                }
                results <- result
            }
        }(item)
    }
    
    // Collect results...
    return nil
}
```

#### 5. Package Organization
- **Keep packages focused** - One clear purpose per package
- **Use internal/ for private code** that shouldn't be imported
- **Avoid circular dependencies**
- **Use meaningful package names** - Short, lowercase, no underscores

```
myproject/
├── cmd/
│   └── myapp/
│       └── main.go
├── internal/
│   ├── domain/
│   │   ├── user.go
│   │   └── order.go
│   ├── repository/
│   │   └── postgres/
│   └── service/
├── pkg/
│   └── api/
└── go.mod
```

#### 6. Context Usage
- **Pass context as first parameter** to functions
- **Don't store context in structs**
- **Use context.TODO()** when context is not yet available
- **Propagate context** through call chains

```go
// Good: Context as first parameter
func FetchUser(ctx context.Context, userID string) (*User, error) {
    // Use ctx for cancellation, timeouts, values
    return repo.FindByID(ctx, userID)
}
```

#### 7. Defer, Panic, Recover
- **Use defer for cleanup** (closing files, unlocking mutexes, etc.)
- **Defer executes in LIFO order**
- **Use panic only for programmer errors**
- **Recover from panics in top-level functions** if necessary

```go
func ProcessFile(filename string) error {
    f, err := os.Open(filename)
    if err != nil {
        return err
    }
    defer f.Close() // Guaranteed to execute
    
    // Process file...
    return nil
}
```

#### 8. Zero Values
- **Design for zero values** - Make zero value useful when possible
- **Use pointers for optional fields**
- **Document when zero value is not valid**

```go
// Good: Zero value is useful
type Config struct {
    Timeout time.Duration // Zero value (0) is valid
    MaxRetries int        // Zero value (0) means no retries
}

// Good: Pointer for optional field
type User struct {
    Name  string
    Email *string // nil means no email provided
}
```

### Code Quality Principles

#### 1. Avoid Default Values - Fail Fast
- **Never use default values to hide errors**
- **Make errors explicit and visible**
- **Fail fast at the entry point** of your system

```go
// Bad: Default value hides error
func GetConfig(key string) string {
    value, err := fetchConfig(key)
    if err != nil {
        return "default" // Silently fails
    }
    return value
}

// Good: Explicit error handling
func GetConfig(key string) (string, error) {
    value, err := fetchConfig(key)
    if err != nil {
        return "", fmt.Errorf("config key %s not found: %w", key, err)
    }
    return value, nil
}
```

#### 2. Consistency Across Code Paths
- **All code paths must be consistent** in their behavior
- **Don't mix error handling strategies**
- **Return the same type across all branches**

```go
// Bad: Inconsistent return types
func ProcessOrder(order Order) interface{} {
    if order.IsValid() {
        return ProcessResult{Success: true}
    }
    return errors.New("invalid order")
}

// Good: Consistent return types
func ProcessOrder(order Order) (*ProcessResult, error) {
    if !order.IsValid() {
        return nil, errors.New("invalid order")
    }
    result := &ProcessResult{Success: true}
    return result, nil
}
```

#### 3. Extract Reusable Functions
- **DRY (Don't Repeat Yourself)**
- **Extract common patterns** into functions
- **Use generics (Go 1.18+)** for type-safe reusable code

```go
// Good: Extracted reusable function
func Map[T, U any](slice []T, fn func(T) U) []U {
    result := make([]U, len(slice))
    for i, v := range slice {
        result[i] = fn(v)
    }
    return result
}

// Usage
names := Map(users, func(u User) string { return u.Name })
```

### Result Pattern in Go

Go doesn't have a built-in Result type, but we can implement it:

```go
// Result represents either a success value or an error
type Result[T any] struct {
    value T
    err   error
}

// Ok creates a successful Result
func Ok[T any](value T) Result[T] {
    return Result[T]{value: value}
}

// Err creates a failed Result
func Err[T any](err error) Result[T] {
    return Result[T]{err: err}
}

// IsOk returns true if the Result is successful
func (r Result[T]) IsOk() bool {
    return r.err == nil
}

// IsErr returns true if the Result is an error
func (r Result[T]) IsErr() bool {
    return r.err != nil
}

// Unwrap returns the value and error
func (r Result[T]) Unwrap() (T, error) {
    return r.value, r.err
}

// Map transforms the value if Ok
func (r Result[T]) Map(fn func(T) T) Result[T] {
    if r.IsErr() {
        return r
    }
    return Ok(fn(r.value))
}

// Usage
func ValidateUser(user User) Result[User] {
    if user.Email == "" {
        return Err[User](errors.New("email is required"))
    }
    return Ok(user)
}
```

---

## Commit

### Git Workflow Rules

**⚠️ CRITICAL: Git Rules**

1. **NEVER use `git stash`** - Always commit or discard changes explicitly
2. **NEVER use `git push`** - User will push manually
3. **NEVER use `git push --force`** - Absolutely forbidden
4. **ALWAYS ask user approval** before committing

### Commit Message Format

Use conventional commits format:

```
<type>(agt): <description>

[optional body]
```

**Types:**
- `feat(agt):` - New feature
- `fix(agt):` - Bug fix
- `refac(agt):` - Refactoring (no functional changes)
- `chore(agt):` - Maintenance tasks
- `doc(agt):` - Documentation
- `test(agt):` - Tests only

**Examples:**
```
feat(agt): add user authentication with JWT
fix(agt): resolve race condition in order processing
refac(agt): extract payment logic into separate package
test(agt): add table-driven tests for validation
```

### Commit Workflow

1. **Run tests** and verify coverage
2. **Review changes** with user
3. **Ask for commit approval**
4. **Create commit** with proper message
5. **Confirm** commit was successful

```bash
# Stage changes
git add .

# Commit with message
git commit -m "feat(agt): implement user registration"

# Verify commit
git log -1
```

---

## Go-Specific Tools

### Build and Run
```bash
# Build
go build -o myapp ./cmd/myapp

# Run
go run ./cmd/myapp

# Install dependencies
go mod tidy
go mod download
```

### Testing
```bash
# Run all tests
go test ./...

# Run tests with coverage
go test ./... -cover

# Generate coverage report
go test ./... -coverprofile=coverage.out
go tool cover -html=coverage.out

# Run specific test
go test -run TestUserValidation ./internal/domain

# Run tests with race detector
go test -race ./...

# Benchmark tests
go test -bench=. ./...
```

### Code Quality
```bash
# Format code
go fmt ./...
gofmt -w .

# Lint code
golangci-lint run

# Vet code (static analysis)
go vet ./...

# Check for common mistakes
staticcheck ./...
```

### Dependencies
```bash
# Add dependency
go get github.com/pkg/errors

# Update dependency
go get -u github.com/pkg/errors

# Remove unused dependencies
go mod tidy

# Vendor dependencies
go mod vendor
```

---

## Go Project Structure

### Standard Layout

```
myproject/
├── cmd/                    # Main applications
│   └── myapp/
│       └── main.go
├── internal/               # Private application code
│   ├── domain/            # Domain models and business logic
│   ├── repository/        # Data access layer
│   ├── service/           # Application services
│   └── handler/           # HTTP handlers
├── pkg/                    # Public libraries
│   └── api/
├── test/                   # Additional test files
├── scripts/                # Build and deployment scripts
├── docs/                   # Documentation
├── go.mod                  # Go module definition
├── go.sum                  # Dependency checksums
├── .gitignore
├── README.md
└── Makefile               # Build automation
```

---

## Additional Resources

### Official Documentation
- [Go Documentation](https://go.dev/doc/)
- [Effective Go](https://go.dev/doc/effective_go)
- [Go Code Review Comments](https://github.com/golang/go/wiki/CodeReviewComments)

### Testing
- [Testing package](https://pkg.go.dev/testing)
- [Testify](https://github.com/stretchr/testify) - Testing toolkit

### Best Practices
- [Go Proverbs](https://go-proverbs.github.io/)
- [Uber Go Style Guide](https://github.com/uber-go/guide/blob/master/style.md)
- [Standard Go Project Layout](https://github.com/golang-standards/project-layout)

---

**Remember**: This file is your primary reference. Read it completely before starting any work on a Go project.
