# Go AI Agents - Quick Reference

**📖 For complete documentation, see [AGENTS_RULES.md](./AGENTS_RULES.md)**

## Essential Workflow

```
Plan → Test → Develop → Test → Commit
```

1. **Plan**: Discuss with user, create todo-list, get approval
2. **Test**: Write failing tests first (TDD)
3. **Develop**: Implement with Go best practices
4. **Test**: Run `go test ./... -cover` (90% minimum)
5. **Commit**: Get user approval, use conventional format

## Go Best Practices

### Error Handling
```go
// Always check errors
result, err := doSomething()
if err != nil {
    return fmt.Errorf("context: %w", err)
}
```

### Interfaces
```go
// Accept interfaces, return structs
type UserRepository interface {
    FindByID(ctx context.Context, id string) (*User, error)
}
```

### Concurrency
```go
// Use context for cancellation
func Process(ctx context.Context, items []Item) error {
    for _, item := range items {
        select {
        case <-ctx.Done():
            return ctx.Err()
        default:
            // Process item
        }
    }
    return nil
}
```

### Struct Methods
```go
// Constructor pattern
func NewUser(name, email string) (*User, error) {
    if name == "" || email == "" {
        return nil, errors.New("name and email required")
    }
    return &User{Name: name, Email: email}, nil
}

// Pointer receiver for mutations
func (u *User) UpdateEmail(email string) error {
    u.Email = email
    return nil
}

// Value receiver for read-only
func (u User) String() string {
    return u.Name
}
```

## Testing

### Table-Driven Tests
```go
func TestValidateEmail(t *testing.T) {
    tests := []struct {
        name    string
        email   string
        wantErr bool
    }{
        {"valid", "test@example.com", false},
        {"empty", "", true},
        {"invalid", "notanemail", true},
    }
    
    for _, tt := range tests {
        t.Run(tt.name, func(t *testing.T) {
            err := ValidateEmail(tt.email)
            if (err != nil) != tt.wantErr {
                t.Errorf("got error %v, wantErr %v", err, tt.wantErr)
            }
        })
    }
}
```

### Run Tests
```bash
# All tests with coverage
go test ./... -v -cover

# Generate coverage report
go test ./... -coverprofile=coverage.out
go tool cover -html=coverage.out

# Race detector
go test -race ./...
```

## Code Quality

### ❌ Don't: Hide Errors with Defaults
```go
func GetConfig(key string) string {
    value, err := fetch(key)
    if err != nil {
        return "default" // BAD: Silent failure
    }
    return value
}
```

### ✅ Do: Explicit Error Handling
```go
func GetConfig(key string) (string, error) {
    value, err := fetch(key)
    if err != nil {
        return "", fmt.Errorf("config %s: %w", key, err)
    }
    return value, nil
}
```

## Git Workflow

### Rules
- ❌ **NEVER** `git stash`
- ❌ **NEVER** `git push` or `--force`
- ✅ **ALWAYS** ask user approval before commit

### Commit Format
```
<type>(agt): <description>

feat(agt): add user registration
fix(agt): resolve race condition
refac(agt): extract payment logic
test(agt): add validation tests
```

## Common Commands

```bash
# Build and run
go build -o myapp ./cmd/myapp
go run ./cmd/myapp

# Testing
go test ./...                    # Run all tests
go test ./... -cover             # With coverage
go test -run TestUser ./...      # Specific test

# Code quality
go fmt ./...                     # Format
go vet ./...                     # Static analysis
golangci-lint run                # Lint

# Dependencies
go mod tidy                      # Clean deps
go get github.com/pkg/errors     # Add dep
go mod vendor                    # Vendor deps
```

## Project Structure

```
myproject/
├── cmd/
│   └── myapp/
│       └── main.go
├── internal/
│   ├── domain/
│   ├── repository/
│   └── service/
├── pkg/
├── go.mod
└── go.sum
```

## Quick Tips

1. **Errors**: Always check, wrap with context
2. **Interfaces**: Small and focused
3. **Context**: First parameter, propagate through calls
4. **Defer**: Use for cleanup (close files, unlock mutexes)
5. **Zero values**: Design for useful defaults
6. **Goroutines**: Use for concurrency, channels for communication
7. **Testing**: Table-driven, subtests with t.Run()

## Resources

- [AGENTS_RULES.md](./AGENTS_RULES.md) - Complete rules
- [CODE_QUALITY_PRINCIPLES.md](./CODE_QUALITY_PRINCIPLES.md) - Quality guidelines
- [CONTRIBUTING.md](./CONTRIBUTING.md) - Contribution guide
- [Effective Go](https://go.dev/doc/effective_go) - Official guide
- [Go Code Review Comments](https://github.com/golang/go/wiki/CodeReviewComments)
