# Code Quality Principles

**⚠️ CRITICAL: Read these principles before code reviews and refactoring**

## 1. Avoid Bad Trade-offs with Default Values

### ❌ **Problem**: Creating default/fallback values to mask missing data

```fsharp
// BAD: Hiding potential issues with defaults
let minimalRatePlan = 
    roomStayPrice.RatePlans
    |> List.tryHead
    |> Option.defaultWith (fun () -> 
        // Creating fake data with many defaults
        { RatePlan.RatePlanCode = None
          Commission = { Percent = 0; StatusType = FullCommission }
          CancelPenalty = { CancelDeadline = None; IsNonRefundable = false; Amount = None }
          NoShowPolicy = None
          RatePlanDescription = RatePlanDescription []
          Taxes = []
          // ... many more defaults
        })
```

### ✅ **Solution**: Fail fast with proper error handling

```fsharp
// GOOD: Explicit error handling, no hidden defaults, when it already exists in the code base
let! ratePlan = 
    roomStayPrice
    |> QuotationRequest.RatePlan.ofRoomStayPrice
    |> Result.mapError ResponseErrorType.DomainErrorType
    |> Async.ret
```

```fsharp
// GOOD: Explicit failure to mock and decide later. Tests may fail but it is a baby step to avoid adding too much complexity at first
let guaranteeForCheck = failwith "TODO(agt): Properly convert GuaranteeTerm to Guarantee using Domain.Guarantee.ofGuaranteeTerm"

// BAD: Hides with a default value trade-offs
// TODO(agt): Properly convert GuaranteeTerm to Guarantee using Domain.Guarantee.ofGuaranteeTerm
let guaranteeForCheck: Gds.Domain.Guarantee option = None
```

### **Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ❌ Hides problems until production
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling
- ✅ Consistent error propagation

## 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution. When you encounter similar scenarios in the codebase, use the same approach to handle them.

### Why it matters:
- ✅ Makes code predictable and easier to understand
- ✅ Reduces cognitive load when reading code
- ✅ Makes refactoring safer
- ✅ Helps identify patterns and abstractions

## 3. Extract Reusable Functions to Modules

Organize code for reuse. When you see repeated logic, extract it into a well-named function in an appropriate module.

### Why it matters:
- ✅ Reduces code duplication
- ✅ Makes changes easier (single point of modification)
- ✅ Improves testability
- ✅ Creates a vocabulary of reusable operations
