# AI Agent Rules for F# Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on F# projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
dotnet.exe test --filter "Category!=End2EndPipelineTests" /p:CollectCoverage=true /p:CoverletOutputFormat=cobertura /p:ExcludeByAttribute="GeneratedCodeAttribute"
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check cobertura XML for new/modified code (minimum 90% coverage)

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Domain-Driven Design & Test Types

**Acceptance Tests (Domain-Focused)**
- **Focus**: Business rules, domain input/output
- **Purpose**: Test business behavior, not implementation details
- **Robustness**: Should survive refactoring (high-level, stable contracts)
- **Vocabulary**: Use business/domain terminology
- **When**: Test features from user/business perspective

**Unit Tests (Technical-Focused)**
- **Focus**: Simple, isolated components (business OR technical)
- **Purpose**: Test individual functions, algorithms, patterns
- **Scope**: Lower-level than acceptance tests

#### Separation of Concerns
- **Business rules** vs **Technical rules** must be clearly separated
- Domain-Driven Design: Start with business problems and vocabulary first
- Technical code (algorithms, patterns) implements the domain model

#### The "Red" Phase Matters
1. **Design First**: The "red" phase validates design at compile time
2. **Top-Down Approach**: When appropriate, implement design by:
   - Defining function signatures with correct input/output types
   - Using business vocabulary in types and function names
   - Mocking dependencies with `NotImplementedException` or `failwith "not yet implemented"`
   - **Code must compile** even with mocks
   - Tests stay red but design is validated
3. **Design Quality Indicator**: If underlying parts are easy to implement, the design is good

#### Make It Work, Then Make It Right
- When implementing features that require refactoring:
  1. Make it work (get tests passing)
  2. Make it right (refactor without changing acceptance tests)
- **Acceptance tests should NOT change during refactoring**
- This proves the refactoring preserves business behavior

### Test Commands

```bash
# Run ALL tests (TDD approach - includes integration tests)
dotnet.exe test

# Run unit tests only (excluding integration/E2E)
dotnet.exe test --filter "Category!=End2End&Category!=IntegrationTest"

# Run specific test project
dotnet.exe test <TestProject>.fsproj

# Run with coverage and generate cobertura XML
dotnet.exe test /p:CollectCoverage=true /p:CoverletOutputFormat=cobertura /p:CoverageThreshold=90

# Run with coverage for specific project
dotnet.exe test tests/DomainTests/DomainTests.fsproj /p:CollectCoverage=true /p:CoverletOutputFormat=cobertura
```

**Important**: Always use `dotnet.exe` even in WSL environment.

---

## Development

### Build Environment

The local devenv is a Windows OS with WSL Ubuntu. It is important to add ".exe" to support Windows tooling integration. All dotnet tools are installed in the host (Windows) outside of WSL.

```bash
dotnet.exe build
dotnet.exe restore
dotnet.exe build --configuration Release
```

### Code Quality Principles

#### 1. Avoid Bad Trade-offs with Default Values

**❌ Problem**: Creating default/fallback values to mask missing data

```fsharp
// BAD: Hiding potential issues with defaults
let minimalRatePlan = 
    roomStayPrice.RatePlans
    |> List.tryHead
    |> Option.defaultWith (fun () -> 
        // Creating fake data with many defaults
        { RatePlan.RatePlanCode = None
          Commission = { Percent = 0; StatusType = FullCommission }
          // ... many more defaults
        })
```

**✅ Solution**: Fail fast with proper error handling

```fsharp
// GOOD: Explicit error handling, no hidden defaults
let! ratePlan = 
    roomStayPrice
    |> QuotationRequest.RatePlan.ofRoomStayPrice
    |> Result.mapError ResponseErrorType.DomainErrorType
    |> Async.ret
```

```fsharp
// GOOD: Explicit failure to mock and decide later
let guaranteeForCheck = failwith "TODO(agt): Properly convert GuaranteeTerm to Guarantee"

// BAD: Hides with a default value trade-offs
// TODO(agt): Properly convert GuaranteeTerm to Guarantee
let guaranteeForCheck: Gds.Domain.Guarantee option = None
```

**Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling

#### 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution

#### 3. Extract Reusable Functions to Modules

Organize for reuse

### TODO Comments

When adding todo in the code base, use this prefix: `//TODO(agt)`

---

## F# Specific Guidelines

### Required Patterns

#### Railway-Oriented Programming
✅ **Use Railway-Oriented Programming** for error handling
- Use `Result<'T, 'Error>` types
- Chain operations with `bind`, `map`
- Keep happy path and error path separate

```fsharp
type Result<'T, 'Error> =
    | Ok of 'T
    | Error of 'Error

let validateOrder order =
    order
    |> validateCustomer
    |> Result.bind validateItems
    |> Result.bind calculateTotal
```

#### Async Workflows
✅ **Use async workflows whenever possible**
- All I/O operations should be async
- Database calls, HTTP requests, file operations
- Use `async { }` computation expressions

```fsharp
let fetchOrder orderId = async {
    let! order = database.GetOrder(orderId)
    let! customer = database.GetCustomer(order.CustomerId)
    return { order with Customer = customer }
}
```

#### Pipe Operator (Preferred)
✅ **Prefer `|>` (pipe operator)** over `>>` (function composition)

```fsharp
// Good - using pipe
let result =
    input
    |> validate
    |> transform
    |> save
```

#### Custom Operators
✅ **Use custom operators for Kleisli composition**
- Example: `>=>` for composing Result-returning functions
- Example: `<!>` for mapping over Result

### F# Anti-Patterns to Avoid

#### ❌ Never Use Mutable State
```fsharp
// BAD - mutable
let mutable counter = 0
counter <- counter + 1

// GOOD - immutable
let counter = 0
let newCounter = counter + 1
```

#### ❌ Avoid Primitive Obsession
✅ **Always wrap primitives in types**

```fsharp
// BAD - primitive obsession
let createOrder (customerId: string) (amount: decimal) = ...

// GOOD - wrapped in types
type CustomerId = CustomerId of string
type Amount = Amount of decimal

let createOrder (customerId: CustomerId) (amount: Amount) = ...
```

#### ❌ No Null Values
✅ **Use `Option` type instead of null**

```fsharp
// BAD - null
let findCustomer id =
    if exists id then customer else null

// GOOD - Option
let findCustomer id : Option<Customer> =
    if exists id then Some customer else None
```

#### ❌ Prefer Result Types Over Exceptions
✅ **Use `Result<'T, 'Error>` instead of throwing exceptions**

```fsharp
// BAD - exceptions
let validateOrder order =
    if order.Items.IsEmpty then
        raise (InvalidOperationException "Order has no items")
    order

// GOOD - Result type
let validateOrder order : Result<Order, string> =
    if order.Items.IsEmpty then
        Error "Order has no items"
    else
        Ok order
```

### Naming Conventions

#### File Naming
- **Style**: `PascalCase.fs`
- **Rule**: File names **must match** the module name inside
- **Examples**:
  - `OrderService.fs` contains `module OrderService`
  - `Customer.fs` contains `module Customer`

#### Test File Naming
- **Format**: `ModuleTests.fs`
- **Examples**:
  - `OrderService.fs` → `OrderServiceTests.fs`
  - `Customer.fs` → `CustomerTests.fs`

#### F# Naming Conventions (Standard)
- **Functions**: `camelCase`
  ```fsharp
  let calculateTotal items = ...
  let processOrder order = ...
  ```

- **Types**: `PascalCase`
  ```fsharp
  type Order = { ... }
  type Customer = { ... }
  ```

- **Modules**: `PascalCase`
  ```fsharp
  module OrderService
  module Domain.Customer
  ```

- **Discriminated Union Cases**: `PascalCase`
  ```fsharp
  type PaymentMethod =
      | CreditCard
      | BankTransfer
      | Cash
  ```

- **Record Fields**: `PascalCase`
  ```fsharp
  type Order = {
      OrderId: string
      CustomerName: string
      TotalAmount: decimal
  }
  ```

### Domain Modeling

⚠️ **Ask user confirmation** when modeling domain types
- Discuss type choices (records, discriminated unions, single-case unions)
- Validate domain vocabulary with user
- Confirm validation and error handling approach

### Dependencies

#### Required Libraries
✅ **FSharp.Data** - Must have
- Type providers for JSON, XML, CSV
- Essential for data access

```fsharp
open FSharp.Data

type Users = JsonProvider<"users.json">
let users = Users.Load("https://api.example.com/users")
```

#### Testing Framework
✅ **xUnit** - Approved testing framework

```fsharp
open Xunit

[<Fact>]
let ``should calculate total correctly`` () =
    let result = calculateTotal [10m; 20m; 30m]
    Assert.Equal(60m, result)
```

---

## Git Workflow

### Git Command Rules

1. ❌ **NEVER** use `git stash` / `git push` or its alias `psh`
2. ❌ **NEVER** use `--force` or `--force-with-lease` options
3. ⚠️ Before running a new git command not listed here, **ask to add it to the allow list**
4. ✅ Use git freely for checking differences (`git diff`, `git log`, `git status`)
5. ✅ Commit only when tests pass with assertions AND code coverage is verified

### Allowed Git Commands

```bash
git status
git diff
git log
git add <files>
git commit -m "message"
# Add more commands here as approved
```

### Commit Message Format

All commit messages must start with one of these prefixes:

- `feat(agt):` - New feature or functionality
- `fix(agt):` - Bug fixes in the codebase
- `refac(agt):` - Refactoring (preparing or finishing a feature)
- `chore(agt):` - Cleanup, removing dead code
- `doc(agt):` - Documentation changes (.md files)
- `test(agt):` - When touching test only

**Examples:**
```
feat(agt): add HotelSearch operation for Sabre connector
fix(agt): correct tax calculation in Amadeus rate mapping
refac(agt): extract common validation logic to shared service
chore(agt): remove unused OTA bindings
doc(agt): update CONTRIBUTING.md with new testing guidelines
```

### Commit Process

1. Run all tests (including integration tests)
2. Verify 90% code coverage minimum
3. **Ask permission to commit**
4. **Wait for user review**
5. Only commit after approval

---

## Project Structure & Organization

### Architecture Principles

- **Preserve existing architecture** in the codebase
- **Propose enhancements** when beneficial by:
  - Asking permission first
  - Providing concrete examples
  - Listing pros and cons
  - Explaining impact on existing code

### File Organization

- **Directory structure**: Organize by layer (e.g., `Domain/`, `Services/`, `Infrastructure/`)
- **File granularity**: One file per feature
- **New files**: Ask user for guidance on where to create new files when the location is unclear

### Project Boundaries

#### Never Modify
❌ **Do not modify or commit:**
- `bin/` - Build output directory
- `obj/` - Intermediate build files
- `*.user` - User-specific settings
- `*.suo` - Solution user options
- `.vs/` - Visual Studio cache
- `coverage.cobertura.xml` - Generated coverage reports
- Any other build artifacts

✅ **Ensure `.gitignore` is configured** to exclude these files

#### Can Modify (After Planning)
✅ **Can be modified** (after todo-list approval):
- `*.fsproj` files - Project files
- Configuration files (`appsettings.json`, `local.settings.json`, etc.)
- All source code files

**Remember**: The planning phase ensures all modifications are discussed before implementation.

---

## Quick Start for New Projects

### 1. Create Solution Structure

```bash
# Create solution
dotnet.exe new sln -n ProjectName

# Create projects
dotnet.exe new classlib -lang F# -o src/Domain
dotnet.exe new classlib -lang F# -o src/Services
dotnet.exe new classlib -lang F# -o src/Infrastructure
dotnet.exe new xunit -lang F# -o tests/DomainTests
dotnet.exe new xunit -lang F# -o tests/ServicesTests

# Add projects to solution
dotnet.exe sln add src/Domain/Domain.fsproj
dotnet.exe sln add src/Services/Services.fsproj
dotnet.exe sln add src/Infrastructure/Infrastructure.fsproj
dotnet.exe sln add tests/DomainTests/DomainTests.fsproj
dotnet.exe sln add tests/ServicesTests/ServicesTests.fsproj

# Add project references
dotnet.exe add tests/DomainTests/DomainTests.fsproj reference src/Domain/Domain.fsproj
dotnet.exe add tests/ServicesTests/ServicesTests.fsproj reference src/Services/Services.fsproj
dotnet.exe add src/Services/Services.fsproj reference src/Domain/Domain.fsproj
```

### 2. Add Required Dependencies

```bash
# Add FSharp.Data (must have)
dotnet.exe add src/Domain/Domain.fsproj package FSharp.Data

# Add coverage tools to test projects
dotnet.exe add tests/DomainTests/DomainTests.fsproj package coverlet.collector
dotnet.exe add tests/DomainTests/DomainTests.fsproj package coverlet.msbuild
```

### 3. Create .gitignore

Ensure `.gitignore` includes:
```
bin/
obj/
*.user
*.suo
.vs/
```

---

## When Stuck or Uncertain

### Decision Matrix

| Situation | Action |
|-----------|--------|
| **Architecture questions** | Ask user with pros/cons example |
| **Domain modeling** | Ask user for confirmation with proposed types |
| **File location unclear** | Ask user where to create new files |
| **New git command needed** | Ask to add to allow list |
| **Design complexity** | Use top-down approach with mocks (`NotImplementedException`) |
| **Test strategy unclear** | Start with acceptance tests (domain level) |
| **Coverage below 90%** | Add more test cases before committing |
| **Tests failing** | Do NOT commit - fix tests first |

### General Principle

**When in doubt, ask the user.** Collaboration is essential.

---

## Summary Checklist

Before any code modification:
- [ ] Have you read the README.md and CONTRIBUTING.md?
- [ ] Have you created and discussed a todo-list with the user?
- [ ] Have you received approval to proceed?

During development:
- [ ] Are you writing tests first (TDD)?
- [ ] Are you using Railway-Oriented Programming for error handling?
- [ ] Are you avoiding mutable state?
- [ ] Are you wrapping primitives in types?
- [ ] Are you using Option instead of null?
- [ ] Are you using Result instead of exceptions?

Before committing:
- [ ] Do all tests pass with meaningful assertions?
- [ ] Is code coverage at least 90%?
- [ ] Have you asked permission to commit?
- [ ] Have you received user approval?
- [ ] Is your commit message properly formatted with (agt) prefix?

---

**Last Updated:** 2025-12-19
**Status:** Complete - F# AI Agent Guidelines
