# Code Quality Principles

**⚠️ CRITICAL: Read these principles before code reviews and refactoring**

## 1. Avoid Bad Trade-offs with Default Values

### ❌ **Problem**: Creating default/fallback values to mask missing data

```csharp
// BAD: Hiding potential issues with defaults
var minimalRatePlan = roomStayPrice.RatePlans
    .FirstOrDefault() ?? new RatePlan
    {
        RatePlanCode = null,
        Commission = new Commission { Percent = 0, StatusType = CommissionStatus.Full },
        CancelPenalty = new CancelPenalty { CancelDeadline = null, IsNonRefundable = false },
        // ... many more defaults
    };
```

### ✅ **Solution**: Fail fast with proper error handling

```csharp
// GOOD: Explicit error handling, no hidden defaults
var ratePlanResult = roomStayPrice.RatePlans
    .FirstOrDefault();

if (ratePlanResult == null)
{
    return Result<RatePlan>.Failure("No rate plan found");
}

return Result<RatePlan>.Success(ratePlanResult);
```

```csharp
// GOOD: Explicit failure to mock and decide later
var guaranteeForCheck = throw new NotImplementedException("TODO(agt): Properly convert GuaranteeTerm to Guarantee");

// BAD: Hides with a default value trade-offs
// TODO(agt): Properly convert GuaranteeTerm to Guarantee
Guarantee? guaranteeForCheck = null;
```

### **Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ❌ Hides problems until production
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling
- ✅ Consistent error propagation

## 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution. When you encounter similar scenarios in the codebase, use the same approach to handle them.

### Why it matters:
- ✅ Makes code predictable and easier to understand
- ✅ Reduces cognitive load when reading code
- ✅ Makes refactoring safer
- ✅ Helps identify patterns and abstractions

## 3. Extract Reusable Functions to Modules

Organize code for reuse. When you see repeated logic, extract it into a well-named method in an appropriate class.

### Why it matters:
- ✅ Reduces code duplication
- ✅ Makes changes easier (single point of modification)
- ✅ Improves testability
- ✅ Creates a vocabulary of reusable operations
