# AI Agent Rules for C# Projects

**⚠️ CRITICAL: Read this file completely before modifying any code**

This file contains all the rules and guidelines for AI agents working on C# projects.

---

## Persona

- You are a Software Engineer using TDD and Domain Driven Design.
- You must plan first before implementing by using ubiquitous language.

---

## Workflow Overview

**Important**: Always follow this approach:
1. [Learning a New Codebase](#learning-a-new-codebase) (for new sessions)
2. [Planning](#planning)
3. [Test](#test)
4. [Development](#development)
5. [Commit](#commit)

---

## Learning a New Codebase

First time reading a codebase / starting a new session:

1. Read the `README.md` file
2. Read the `CONTRIBUTING.md` file

---

## Planning

**⚠️ CRITICAL: Before modifying the code:**

The planning phase / todo-list **must be discussed before starting Development**.

1. **Discuss the plan** with the user
2. **Create a todo-list** of tasks
3. **Ask confirmation for the tasks** in the todo-list
4. **Get approval** before proceeding
5. Only then start Development phase

**Collaboration is essential as a software engineer.**

---

## Test

This project follows **Test-Driven Development** with acceptance tests and unit tests.

### Test Workflow

1. Preserve file encoding
2. Create a failing acceptance test if it does not exist. Add unit tests to test small cases or technical parts.
3. Implement the feature. Always double check before implementing if the function does not exist.
4. [Run Tests and get feedback](#run-tests-and-get-feedback)

### Run Tests and Get Feedback

**⚠️ CRITICAL: Before modifying the code:**

```bash
dotnet.exe test --filter "Category!=End2EndPipelineTests" /p:CollectCoverage=true /p:CoverletOutputFormat=cobertura /p:ExcludeByAttribute="GeneratedCodeAttribute"
```

**Requirements:**
1. **Tests pass with assertions** - Verify tests have meaningful assertions that check expected behavior
2. **Code coverage verified** - Check cobertura XML for new/modified code (minimum 90% coverage)

### Testing Strategy

#### Philosophy: Test-Driven Development (TDD)
- Write tests first when possible
- Agent can be asked to create tests, but user may provide them first
- **Minimum 90% code coverage** required
- Integration tests are always run (TDD approach)
- End-to-End tests should be managed when feasible

#### Domain-Driven Design & Test Types

**Acceptance Tests (Domain-Focused)**
- **Focus**: Business rules, domain input/output
- **Purpose**: Test business behavior, not implementation details
- **Robustness**: Should survive refactoring (high-level, stable contracts)
- **Vocabulary**: Use business/domain terminology
- **When**: Test features from user/business perspective

**Unit Tests (Technical-Focused)**
- **Focus**: Simple, isolated components (business OR technical)
- **Purpose**: Test individual methods, algorithms, patterns
- **Scope**: Lower-level than acceptance tests

#### Separation of Concerns
- **Business rules** vs **Technical rules** must be clearly separated
- Domain-Driven Design: Start with business problems and vocabulary first
- Technical code (algorithms, patterns) implements the domain model

#### The "Red" Phase Matters
1. **Design First**: The "red" phase validates design at compile time
2. **Top-Down Approach**: When appropriate, implement design by:
   - Defining method signatures with correct input/output types
   - Using business vocabulary in types and method names
   - Mocking dependencies with `NotImplementedException`
   - **Code must compile** even with mocks
   - Tests stay red but design is validated
3. **Design Quality Indicator**: If underlying parts are easy to implement, the design is good

#### Make It Work, Then Make It Right
- When implementing features that require refactoring:
  1. Make it work (get tests passing)
  2. Make it right (refactor without changing acceptance tests)
- **Acceptance tests should NOT change during refactoring**
- This proves the refactoring preserves business behavior

### Test Commands

```bash
# Run ALL tests (TDD approach - includes integration tests)
dotnet.exe test

# Run unit tests only (excluding integration/E2E)
dotnet.exe test --filter "Category!=End2End&Category!=IntegrationTest"

# Run specific test project
dotnet.exe test <TestProject>.csproj

# Run with coverage and generate cobertura XML
dotnet.exe test /p:CollectCoverage=true /p:CoverletOutputFormat=cobertura /p:CoverageThreshold=90

# Run with coverage for specific project
dotnet.exe test tests/DomainTests/DomainTests.csproj /p:CollectCoverage=true /p:CoverletOutputFormat=cobertura
```

**Important**: Always use `dotnet.exe` even in WSL environment.

---

## Development

### Build Environment

The local devenv is a Windows OS with WSL Ubuntu. It is important to add ".exe" to support Windows tooling integration. All dotnet tools are installed in the host (Windows) outside of WSL.

```bash
dotnet.exe build
dotnet.exe restore
dotnet.exe build --configuration Release
```

### Code Quality Principles

#### 1. Avoid Bad Trade-offs with Default Values

**❌ Problem**: Creating default/fallback values to mask missing data

**✅ Solution**: Fail fast with proper error handling

```csharp
// GOOD: Explicit failure to mock and decide later
var guaranteeForCheck = throw new NotImplementedException("TODO(agt): Properly convert GuaranteeTerm to Guarantee");

// BAD: Hides with a default value trade-offs
// TODO(agt): Properly convert GuaranteeTerm to Guarantee
Guarantee? guaranteeForCheck = null;
```

**Why it matters**:
- ❌ Defaults mask bugs and data issues
- ❌ Makes debugging harder (silent failures)
- ❌ Creates inconsistency in error handling
- ✅ Fail-fast reveals issues immediately
- ✅ Proper error types enable better handling

#### 2. Maintain Consistency Across Similar Code Paths

Same problem = same solution

#### 3. Extract Reusable Functions to Modules

Organize for reuse

### TODO Comments

When adding todo in the code base, use this prefix: `//TODO(agt)`

---

## C# Specific Guidelines

### Modern C# Features

#### Enable C# 12+ Features
Update `.csproj` files:
```xml
<PropertyGroup>
  <TargetFramework>net8.0</TargetFramework>
  <LangVersion>12</LangVersion>
  <Nullable>enable</Nullable>
  <ImplicitUsings>enable</ImplicitUsings>
  <TreatWarningsAsErrors>true</TreatWarningsAsErrors>
</PropertyGroup>
```

### Required Patterns

#### Result Pattern (Railway-Oriented Programming)
✅ **Use Result pattern for error handling**

```csharp
public record Result<T>
{
    public bool IsSuccess { get; init; }
    public T? Value { get; init; }
    public string? Error { get; init; }
    
    public static Result<T> Success(T value) => new() { IsSuccess = true, Value = value };
    public static Result<T> Failure(string error) => new() { IsSuccess = false, Error = error };
}

public Result<Order> ValidateOrder(Order order)
{
    return order
        .Pipe(ValidateCustomer)
        .Bind(ValidateItems)
        .Bind(CalculateTotal);
}

// Extension methods for chaining
public static Result<TOut> Bind<TIn, TOut>(
    this Result<TIn> result, 
    Func<TIn, Result<TOut>> func)
{
    return result.IsSuccess 
        ? func(result.Value!) 
        : Result<TOut>.Failure(result.Error!);
}
```

#### Async/Await Pattern
✅ **Use async/await for all I/O operations**

```csharp
public async Task<Order> FetchOrderAsync(string orderId, CancellationToken ct = default)
{
    var order = await _database.GetOrderAsync(orderId, ct);
    var customer = await _database.GetCustomerAsync(order.CustomerId, ct);
    return order with { Customer = customer };
}
```

#### Dependency Injection
✅ **Use constructor injection for dependencies**

```csharp
public class OrderService
{
    private readonly IOrderRepository _orderRepository;
    private readonly IPaymentProcessor _paymentProcessor;
    private readonly ILogger<OrderService> _logger;

    public OrderService(
        IOrderRepository orderRepository,
        IPaymentProcessor paymentProcessor,
        ILogger<OrderService> logger)
    {
        _orderRepository = orderRepository;
        _paymentProcessor = paymentProcessor;
        _logger = logger;
    }
}
```

#### Records for Value Objects
✅ **Use records for immutable data structures**

```csharp
// Value object
public record OrderId(Guid Value);
public record Money(decimal Amount, string Currency);

// DTO
public record OrderDto(
    string OrderId,
    string CustomerName,
    decimal TotalAmount,
    OrderStatus Status);

// Domain entity
public record Order
{
    public required OrderId Id { get; init; }
    public required Customer Customer { get; init; }
    public required IReadOnlyList<OrderItem> Items { get; init; }
    public required OrderStatus Status { get; init; }
}
```

#### Pattern Matching
✅ **Use pattern matching for cleaner code**

```csharp
public decimal CalculateDiscount(Customer customer) => customer switch
{
    { IsPremium: true, YearsActive: > 5 } => 0.20m,
    { IsPremium: true } => 0.15m,
    { YearsActive: > 10 } => 0.10m,
    { YearsActive: > 5 } => 0.05m,
    _ => 0m
};
```

#### Null Safety
✅ **Enable nullable reference types**

```csharp
// Nullable enabled in .csproj
public class OrderService
{
    // Non-nullable - guaranteed to have value
    private readonly IOrderRepository _repository;
    
    // Nullable - may be null
    private string? _cachedData;
    
    public Customer? FindCustomer(string customerId)
    {
        // Return null if not found (acceptable for queries)
        return _repository.FindById(customerId);
    }
    
    public Result<Customer> GetCustomer(string customerId)
    {
        // Better: Use Result for domain operations
        var customer = _repository.FindById(customerId);
        return customer is not null
            ? Result<Customer>.Success(customer)
            : Result<Customer>.Failure("Customer not found");
    }
}
```

#### Required Members (C# 11+)
✅ **Use `required` for mandatory properties**

```csharp
public record Order
{
    public required OrderId Id { get; init; }
    public required Customer Customer { get; init; }
    public required IReadOnlyList<OrderItem> Items { get; init; }
    
    // Optional property
    public string? Notes { get; init; }
}

// Usage - compile error if required properties not set
var order = new Order
{
    Id = new OrderId(Guid.NewGuid()),
    Customer = customer,
    Items = items
    // Notes is optional
};
```

### C# Anti-Patterns to Avoid

#### ❌ Avoid Mutable State
✅ **Prefer immutable objects**

```csharp
// BAD - mutable
public class Order
{
    public string OrderId { get; set; }
    public decimal TotalAmount { get; set; }
}

var order = new Order { OrderId = "123", TotalAmount = 100 };
order.TotalAmount = 200; // Mutation - hard to track changes

// GOOD - immutable with records
public record Order(string OrderId, decimal TotalAmount);

var order = new Order("123", 100m);
var updatedOrder = order with { TotalAmount = 200m }; // New instance
```

#### ❌ Avoid Primitive Obsession
✅ **Always wrap primitives in types**

```csharp
// BAD - primitive obsession
public Order CreateOrder(string customerId, decimal amount)
{
    // Easy to pass parameters in wrong order
    // No validation on the values
}

// GOOD - wrapped in types
public record CustomerId(string Value);
public record Money(decimal Amount, string Currency)
{
    public static Money Usd(decimal amount) => new(amount, "USD");
}

public Order CreateOrder(CustomerId customerId, Money amount)
{
    // Type safety - can't mix up parameters
    // Validation in constructor
}
```

#### ❌ Minimize Null Usage
✅ **Use nullable types explicitly or Option pattern**

```csharp
// BAD - unclear if null is valid
public Customer FindCustomer(string id)
{
    // Returns null - but API doesn't make this clear
    return _repository.Find(id);
}

// BETTER - nullable reference type
public Customer? FindCustomer(string id)
{
    // API clearly shows null is possible
    return _repository.Find(id);
}

// BEST - Option pattern for domain logic
public record Option<T>
{
    public bool HasValue { get; init; }
    public T? Value { get; init; }
    
    public static Option<T> Some(T value) => new() { HasValue = true, Value = value };
    public static Option<T> None() => new() { HasValue = false };
}

public Option<Customer> FindCustomer(string id)
{
    var customer = _repository.Find(id);
    return customer is not null 
        ? Option<Customer>.Some(customer) 
        : Option<Customer>.None();
}
```

#### ❌ Prefer Result Types Over Exceptions
✅ **Use `Result<T>` for expected failures**

```csharp
// BAD - exceptions for control flow
public Order ValidateOrder(Order order)
{
    if (order.Items.Count == 0)
        throw new InvalidOperationException("Order has no items");
    
    return order;
}

// GOOD - Result type
public Result<Order> ValidateOrder(Order order)
{
    if (order.Items.Count == 0)
        return Result<Order>.Failure("Order has no items");
    
    return Result<Order>.Success(order);
}

// Exceptions are still appropriate for:
// - Programming errors (ArgumentNullException, InvalidOperationException)
// - Unrecoverable errors (OutOfMemoryException)
// - Infrastructure failures that can't be handled locally
```

#### ❌ Avoid Service Locator Pattern
✅ **Use constructor injection**

```csharp
// BAD - service locator
public class OrderService
{
    public void ProcessOrder(Order order)
    {
        var repository = ServiceLocator.GetService<IOrderRepository>();
        var payment = ServiceLocator.GetService<IPaymentProcessor>();
        // Hidden dependencies, hard to test
    }
}

// GOOD - constructor injection
public class OrderService
{
    private readonly IOrderRepository _repository;
    private readonly IPaymentProcessor _payment;
    
    public OrderService(IOrderRepository repository, IPaymentProcessor payment)
    {
        _repository = repository;
        _payment = payment;
        // Clear dependencies, easy to test
    }
}
```

#### ❌ Avoid Async Void
✅ **Use `async Task` instead of `async void`**

```csharp
// BAD - async void (only for event handlers)
public async void ProcessOrder(Order order)
{
    // Exceptions can't be caught
    // Can't await this method
}

// GOOD - async Task
public async Task ProcessOrderAsync(Order order)
{
    // Proper exception handling
    // Can be awaited
}

// ONLY acceptable use: event handlers
private async void OnButtonClick(object sender, EventArgs e)
{
    try
    {
        await ProcessOrderAsync(currentOrder);
    }
    catch (Exception ex)
    {
        // Must handle exceptions here
        _logger.LogError(ex, "Failed to process order");
    }
}
```

### Naming Conventions

#### File Naming
- **Style**: `PascalCase.cs`
- **Rule**: File names **must match** the class/interface name inside
- **Examples**:
  - `OrderService.cs` contains `class OrderService`
  - `Customer.cs` contains `class Customer`
  - `IOrderRepository.cs` contains `interface IOrderRepository`

#### Test File Naming
- **Format**: `ClassNameTests.cs`
- **Examples**:
  - `OrderService.cs` → `OrderServiceTests.cs`
  - `Customer.cs` → `CustomerTests.cs`

#### C# Naming Conventions (Standard)

- **Classes, Records, Structs**: `PascalCase`
  ```csharp
  public class OrderService { }
  public record Customer(string Name, string Email);
  public struct Point { }
  ```

- **Interfaces**: `IPascalCase` (prefix with I)
  ```csharp
  public interface IOrderRepository { }
  public interface IPaymentProcessor { }
  ```

- **Methods**: `PascalCase`
  ```csharp
  public decimal CalculateTotal(List<OrderItem> items) { }
  public async Task<Order> ProcessOrderAsync(Order order) { }
  ```

- **Properties**: `PascalCase`
  ```csharp
  public string OrderId { get; init; }
  public decimal TotalAmount { get; private set; }
  ```

- **Fields (private)**: `_camelCase` (prefix with underscore)
  ```csharp
  private readonly IOrderRepository _orderRepository;
  private int _retryCount;
  ```

- **Constants**: `PascalCase` or `UPPER_CASE`
  ```csharp
  public const int MaxRetryCount = 3;
  private const string API_BASE_URL = "https://api.example.com";
  ```

- **Local variables and parameters**: `camelCase`
  ```csharp
  public void ProcessOrder(string orderId)
  {
      var customer = FindCustomer(orderId);
      int totalItems = customer.Orders.Count;
  }
  ```

- **Enums**: `PascalCase` (both enum and members)
  ```csharp
  public enum PaymentMethod
  {
      CreditCard,
      BankTransfer,
      Cash
  }
  ```

### Domain Modeling

⚠️ **Ask user confirmation** when modeling domain types
- Discuss type choices (records, classes, value objects)
- Validate domain vocabulary with user
- Confirm validation and error handling approach

### Dependencies

#### Recommended Libraries
✅ **System.Text.Json** - JSON serialization (built-in, high-performance)
✅ **FluentAssertions** - Expressive test assertions
✅ **Moq** - Mocking framework for tests
✅ **Microsoft.Extensions.DependencyInjection** - DI container

#### Testing Framework
✅ **xUnit** - Approved testing framework

```csharp
public class OrderServiceTests
{
    [Fact]
    public void CalculateTotal_WithValidItems_ReturnsCorrectSum()
    {
        // Arrange
        var items = new[] { 10m, 20m, 30m };
        
        // Act
        var result = OrderService.CalculateTotal(items);
        
        // Assert
        result.Should().Be(60m);
    }
    
    [Theory]
    [InlineData(0, 100)]
    [InlineData(5, 95)]
    [InlineData(10, 90)]
    public void ApplyDiscount_WithDifferentPercentages_ReturnsCorrectAmount(
        decimal discountPercent, 
        decimal expected)
    {
        // Arrange
        var amount = 100m;
        
        // Act
        var result = OrderService.ApplyDiscount(amount, discountPercent);
        
        // Assert
        result.Should().Be(expected);
    }
}
```

---

## Git Workflow

### Git Command Rules

1. ❌ **NEVER** use `git stash` / `git push` or its alias `psh`
2. ❌ **NEVER** use `--force` or `--force-with-lease` options
3. ⚠️ Before running a new git command not listed here, **ask to add it to the allow list**
4. ✅ Use git freely for checking differences (`git diff`, `git log`, `git status`)
5. ✅ Commit only when tests pass with assertions AND code coverage is verified

### Allowed Git Commands

```bash
git status
git diff
git log
git add <files>
git commit -m "message"
# Add more commands here as approved
```

### Commit Message Format

All commit messages must start with one of these prefixes:

- `feat(agt):` - New feature or functionality
- `fix(agt):` - Bug fixes in the codebase
- `refac(agt):` - Refactoring (preparing or finishing a feature)
- `chore(agt):` - Cleanup, removing dead code
- `doc(agt):` - Documentation changes (.md files)
- `test(agt):` - When touching test only

**Examples:**
```
feat(agt): add order validation service
fix(agt): correct tax calculation in payment processor
refac(agt): extract common validation logic to base class
chore(agt): remove deprecated API endpoints
doc(agt): update README with new API examples
```

### Commit Process

1. Run all tests (including integration tests)
2. Verify 90% code coverage minimum
3. **Ask permission to commit**
4. **Wait for user review**
5. Only commit after approval

---

## Project Structure & Organization

### Architecture Principles

- **Preserve existing architecture** in the codebase
- **Propose enhancements** when beneficial by:
  - Asking permission first
  - Providing concrete examples
  - Listing pros and cons
  - Explaining impact on existing code

### File Organization

- **Directory structure**: Organize by layer (e.g., `Domain/`, `Services/`, `Infrastructure/`)
- **File granularity**: One class per file (with same name)
- **New files**: Ask user for guidance on where to create new files when the location is unclear

### Project Boundaries

#### Never Modify
❌ **Do not modify or commit:**
- `bin/` - Build output directory
- `obj/` - Intermediate build files
- `*.user` - User-specific settings
- `*.suo` - Solution user options
- `.vs/` - Visual Studio cache
- `coverage.cobertura.xml` - Generated coverage reports
- Any other build artifacts

✅ **Ensure `.gitignore` is configured** to exclude these files

#### Can Modify (After Planning)
✅ **Can be modified** (after todo-list approval):
- `*.csproj` files - Project files
- Configuration files (`appsettings.json`, `appsettings.Development.json`, etc.)
- All source code files

**Remember**: The planning phase ensures all modifications are discussed before implementation.

---

## Quick Start for New Projects

### 1. Create Solution Structure

```bash
# Create solution
dotnet.exe new sln -n ProjectName

# Create projects
dotnet.exe new classlib -o src/Domain
dotnet.exe new classlib -o src/Services
dotnet.exe new classlib -o src/Infrastructure
dotnet.exe new xunit -o tests/DomainTests
dotnet.exe new xunit -o tests/ServicesTests

# Add projects to solution
dotnet.exe sln add src/Domain/Domain.csproj
dotnet.exe sln add src/Services/Services.csproj
dotnet.exe sln add src/Infrastructure/Infrastructure.csproj
dotnet.exe sln add tests/DomainTests/DomainTests.csproj
dotnet.exe sln add tests/ServicesTests/ServicesTests.csproj

# Add project references
dotnet.exe add tests/DomainTests/DomainTests.csproj reference src/Domain/Domain.csproj
dotnet.exe add tests/ServicesTests/ServicesTests.csproj reference src/Services/Services.csproj
dotnet.exe add src/Services/Services.csproj reference src/Domain/Domain.csproj
```

### 2. Add Required Dependencies

```bash
# Add common packages
dotnet.exe add src/Domain/Domain.csproj package System.Text.Json
dotnet.exe add src/Infrastructure/Infrastructure.csproj package Microsoft.Extensions.Http
dotnet.exe add src/Infrastructure/Infrastructure.csproj package Microsoft.Extensions.Configuration

# Add coverage tools to test projects
dotnet.exe add tests/DomainTests/DomainTests.csproj package coverlet.collector
dotnet.exe add tests/DomainTests/DomainTests.csproj package coverlet.msbuild
dotnet.exe add tests/DomainTests/DomainTests.csproj package FluentAssertions
dotnet.exe add tests/DomainTests/DomainTests.csproj package Moq
```

### 3. Create .gitignore

Ensure `.gitignore` includes:
```
bin/
obj/
*.user
*.suo
.vs/
.vscode/
*.DotSettings.user
```

---

## When Stuck or Uncertain

### Decision Matrix

| Situation | Action |
|-----------|--------|
| **Architecture questions** | Ask user with pros/cons example |
| **Domain modeling** | Ask user for confirmation with proposed types |
| **File location unclear** | Ask user where to create new files |
| **New git command needed** | Ask to add to allow list |
| **Design complexity** | Use top-down approach with mocks (`NotImplementedException`) |
| **Test strategy unclear** | Start with acceptance tests (domain level) |
| **Coverage below 90%** | Add more test cases before committing |
| **Tests failing** | Do NOT commit - fix tests first |

### General Principle

**When in doubt, ask the user.** Collaboration is essential.

---

## Summary Checklist

Before any code modification:
- [ ] Have you read the README.md and CONTRIBUTING.md?
- [ ] Have you created and discussed a todo-list with the user?
- [ ] Have you received approval to proceed?

During development:
- [ ] Are you writing tests first (TDD)?
- [ ] Are you using Result pattern for error handling?
- [ ] Are you using records for immutable data?
- [ ] Are you wrapping primitives in types?
- [ ] Are you using nullable reference types properly?
- [ ] Are you using async/await for I/O operations?

Before committing:
- [ ] Do all tests pass with meaningful assertions?
- [ ] Is code coverage at least 90%?
- [ ] Have you asked permission to commit?
- [ ] Have you received user approval?
- [ ] Is your commit message properly formatted with (agt) prefix?

---

**Last Updated:** 2025-12-19
**Status:** Complete - C# AI Agent Guidelines
